/*-----------------------------------------------------------------------------
 * EthernetServices.h       
 * Copyright                acontis technologies GmbH, Weingarten, Germany
 * Response                 Stefan Zintgraf
 * Description              
 *---------------------------------------------------------------------------*/

#ifndef INC_ETHERNETSERVICES
#define INC_ETHERNETSERVICES


/*-INCLUDES------------------------------------------------------------------*/
#ifndef INC_ECINTERFACECOMMON
#include "EcInterfaceCommon.h"
#endif

/*-PACK SETTINGS-------------------------------------------------------------*/
#if defined  __GNUC__   /* GNU */

#define EC_PACKED1 __attribute__((aligned(1), packed))

#if (defined __RCX__)
#pragma pack (1)
#endif

#elif defined _DIAB_TOOL

#define EC_PACKED1 __attribute__((aligned(1), packed))

#elif defined _MSC_VER  /* MICROSOFT */

#define EC_PACKED1
#pragma pack (push, 1)
#pragma warning(disable:4514)

#elif defined __MET__

#define EC_PACKED1
/*#pragma pack (push, 1)*/
#pragma Push_align_members(1)

#else
#error Please check pack settings for this compiler!!!
#endif



/*---------------------------------------------------------------------------*/
/* Class D IP addresses, in the range 224.0.0.0 to 239.255.255.255 are called multicast addresses */
/*---------------------------------------------------------------------------*/

/*-DEFINES/MACROS------------------------------------------------------------*/
#ifndef SWAPWORD
    #define SWAPWORD(w)                         (EC_T_WORD)(((((EC_T_WORD)w)&0xff) << 8) | ((((EC_T_WORD)w)&0xff00) >> 8))
#endif


#ifdef EC_BIG_ENDIAN
#define ETHERNET_FRAME_TYPE_IP          0x0008
#define ETHERNET_FRAME_TYPE_ARP1        0x0608
#define ETHERNET_FRAME_TYPE_ARP2        0x0708
#define ETHERNET_FRAME_TYPE_RARP        0x3580
#define ETHERNET_FRAME_TYPE_VLAN        0x0081
#define ETHERNET_FRAME_TYPE_SNMP        0x4C81
#define ETHERNET_FRAME_TYPE_LOOP        0x0090
#define ETHERNET_FRAME_TYPE_BKHF        0xA488
#define ETHERNET_FRAME_TYPE_PROFINET    0x9288
#else
#define ETHERNET_FRAME_TYPE_IP          0x0800
#define ETHERNET_FRAME_TYPE_ARP1        0x0806
#define ETHERNET_FRAME_TYPE_ARP2        0x0807
#define ETHERNET_FRAME_TYPE_RARP        0x8035
#define ETHERNET_FRAME_TYPE_VLAN        0x8100
#define ETHERNET_FRAME_TYPE_SNMP        0x814C
#define ETHERNET_FRAME_TYPE_LOOP        0x9000
#define ETHERNET_FRAME_TYPE_BKHF        0x88A4
#define ETHERNET_FRAME_TYPE_PROFINET    0x8892
#endif

#define ETHERNET_FRAME_TYPE_IP_SW       SWAPWORD(ETHERNET_FRAME_TYPE_IP)
#define ETHERNET_FRAME_TYPE_ARP1_SW     SWAPWORD(ETHERNET_FRAME_TYPE_ARP1)
#define ETHERNET_FRAME_TYPE_ARP2_SW     SWAPWORD(ETHERNET_FRAME_TYPE_ARP2)
#define ETHERNET_FRAME_TYPE_RARP_SW     SWAPWORD(ETHERNET_FRAME_TYPE_RARP)
#define ETHERNET_FRAME_TYPE_VLAN_SW     SWAPWORD(ETHERNET_FRAME_TYPE_VLAN)  
#define ETHERNET_FRAME_TYPE_SNMP_SW     SWAPWORD(ETHERNET_FRAME_TYPE_SNMP)
#define ETHERNET_FRAME_TYPE_LOOP_SW     SWAPWORD(ETHERNET_FRAME_TYPE_LOOP)
#define ETHERNET_FRAME_TYPE_BKHF_SW     SWAPWORD(ETHERNET_FRAME_TYPE_BKHF)
#define ETHERNET_FRAME_TYPE_PROFINET_SW SWAPWORD(ETHERNET_FRAME_TYPE_PROFINET)

#define ETHERNET_MAX_FRAME_LEN                  1514
#define ETHERNET_MAX_VLAN_FRAME_LEN             1518
#define ETHERNET_MAX_FRAMEBUF_LEN               1536

#define ETHERNET_FRAME_DURATION_MBAUD(l, m)     ((8*((EC_MAX(60, (l)))+24))/(m))
#define ETHERNET_FRAME_LENGTH_MBAUD(t, m)       ((((t)*(m))/8) - 24)

#define ETHERNET_FRAME_DURATION_100(l)          ETHERNET_FRAME_DURATION_MBAUD((l), 100)
#define ETHERNET_FRAME_LENGTH_100(t)            ETHERNET_FRAME_LENGTH_MBAUD((t), 100)

#define ETHERNET_BYTE_DURATION_100(l)           ((2*(l))/25)
#define ETHERNET_TIME_LENGTH_100(t)             (((t)*25)/2)



/*---------------------------------------------------------------------------*/

/*-TYPEDEFS/ENUMS------------------------------------------------------------*/
typedef struct TETHERNET_ADDRESS
{
    EC_T_BYTE b[6];

#ifdef __cplusplus
    EC_T_INT operator% (EC_T_INT hashSize)  /* for calculating the hashvalue */
    {
        return (EC_GETDWORD(b) % hashSize);
    }
    EC_T_BOOL operator== ( const TETHERNET_ADDRESS &addr ) const
    {
        return ( OsMemcmp( b, addr.b, sizeof(TETHERNET_ADDRESS) ) == 0 );
    }
    EC_T_BOOL operator!= ( const TETHERNET_ADDRESS &addr ) const
    {
        return ( OsMemcmp( b, addr.b, sizeof(TETHERNET_ADDRESS) ) != 0 );
    }
    EC_T_VOID operator= ( const TETHERNET_ADDRESS &addr )
    { 
        OsMemcpy( &b, &addr.b, sizeof(b) );
    }
#endif  /* __cplusplus */
} EC_PACKED1 ETHERNET_ADDRESS, *PETHERNET_ADDRESS;

#define ETHERNET_ADDRESS_LEN    sizeof(ETHERNET_ADDRESS)
static const   ETHERNET_ADDRESS        BroadcastEthernetAddress        ={{0xFF,0xFF,0xFF,0xFF,0xFF,0xFF}};
static const   ETHERNET_ADDRESS        FirstMulticastEthernetAddress   ={{0x01,0x00,0x5E,0x00,0x00,0x00}};
static const   ETHERNET_ADDRESS        NullEthernetAddress             ={{0x00,0x00,0x00,0x00,0x00,0x00}};

/*---------------------------------------------------------------------------*/
typedef struct TETHERNET_FRAME 
{
    ETHERNET_ADDRESS    Destination;   /* 0 */
    ETHERNET_ADDRESS    Source;        /* 6 */
    EC_T_WORD           __FrameType;   /*12 */  /* in host-order */
} EC_PACKED1 ETHERNET_FRAME, *PETHERNET_FRAME;

#define EC_ETHFRM_GET_FRAMETYPE(p)        EC_GETWORD( (((EC_T_BYTE*)(p))+12) )

#define EC_ETHFRM_SET_FRAMETYPE(p, wVal)  EC_SETWORD( (((EC_T_BYTE*)(p))+12) , (wVal))

#define EC_ETHFRM_SET_RETRYINDEX(p, byVal)  (*(((EC_T_BYTE*)(p))+(6+1))) = (EC_LOBYTE((byVal)))
#define EC_ETHFRM_GET_RETRYINDEX(p)         EC_LOBYTE((*(((EC_T_BYTE*)(p))+(6+1))))

#define EC_ETHFRM_ADD_RETRYINDEX(p, byVal)  EC_ETHFRM_SET_RETRYINDEX((p), EC_LOBYTE((EC_ETHFRM_GET_RETRYINDEX(p)+(byVal))))
#define EC_ETHFRM_SET_RETRYMACIDX(p, abyMac, byVal ) EC_ETHFRM_SET_RETRYINDEX((p), EC_LOBYTE(((abyMac[1])+(byVal))))

            
#define ETHERNET_FRAME_LEN      sizeof(ETHERNET_FRAME)
#define ETHERNET_FRAMETYPE_LEN  sizeof(EC_T_WORD)

#define EC_VLANHDR_OFFS_VLANTYPE    ((EC_T_BYTE)0)
#define EC_VLANHDR_OFFS_VLANIDH     ((EC_T_BYTE)2)
/*---------------------------------------------------------------------------*/
typedef struct TETYPE_VLAN_HEADER
{
#ifdef EC_BIG_ENDIAN
    /* TODO: currently not used and not implemented for big endianness */
    EC_T_WORD   __VLanIdL       : 8;    /* 3 */        
    EC_T_WORD   __Priority      : 3;    /* 2.5 */    /* 0 = lowest, 7 highest */
    EC_T_WORD   Reserved        : 1;    /* 2.4 */        
    EC_T_WORD   __VLanIdH       : 4;    /* 2.0 */        
    EC_T_WORD   __VLanType;             /*  0 */    /* 0x8100 in host-order (0x0081) */
#else
    EC_T_WORD   __VLanType;             /*  0 */    /* 0x8100 in host-order (0x0081) */
    EC_T_WORD   __VLanIdH       : 4;    /* 2.0 */        
    EC_T_WORD   Reserved        : 1;    /* 2.4 */        
    EC_T_WORD   __Priority      : 3;    /* 2.5 */    /* 0 = lowest, 7 highest */
    EC_T_WORD   __VLanIdL       : 8;    /* 3 */        
#endif
} EC_PACKED1 ETYPE_VLAN_HEADER, *PETYPE_VLAN_HEADER;
#define ETYPE_VLAN_HEADER_LEN   sizeof(ETYPE_VLAN_HEADER)

#define EC_AL_GET_VLANHDR_IDH(p)            ((p)->__VLanIdH)
#define EC_AL_GET_VLANHDR_IDL(p)            ((p)->__VLanIdL)
#define EC_AL_GET_VLANHDR_Prio(p)           ((p)->__Priority)
#define EC_AL_GET_VLANHDR_Type(p)           ((p)->__VLanType)
#define EC_AL_GET_VLANHDR_ID(p)             ((EC_T_WORD)((((p)->__VLanIdH)<<8) | ((p)->__VLanIdL)))
                                            
#define EC_AL_SET_VLANHDR_IDH(p, wVal)      ((p)->__VLanIdH)    =   (wVal&0xf)
#define EC_AL_SET_VLANHDR_IDL(p, wVal)      ((p)->__VLanIdL)    =   (wVal&0xff)
#define EC_AL_SET_VLANHDR_Prio(p, byVal)    ((p)->__Priority)   =   (byVal)
#define EC_AL_SET_VLANHDR_Type(p, wVal)     ((p)->__VLanType)   =   (wVal)
#define EC_AL_SET_VLANHDR_ID(p, wVal)       EC_AL_SET_VLANHDR_IDH((p), (((wVal)>>8)&0xff)); EC_AL_SET_VLANHDR_IDL((p), (((wVal)>>0)&0xf))
         
#if !(defined WITHALIGNMENT)
#define EC_GET_VLANHDR_IDH(p)            EC_AL_GET_VLANHDR_IDH(p)
#define EC_GET_VLANHDR_IDL(p)            EC_AL_GET_VLANHDR_IDL(p)
#define EC_GET_VLANHDR_Prio(p)           EC_AL_GET_VLANHDR_Prio(p)
#define EC_GET_VLANHDR_Type(p)           EC_AL_GET_VLANHDR_Type(p)
#define EC_GET_VLANHDR_ID(p)             EC_AL_GET_VLANHDR_ID(p)

#define EC_SET_VLANHDR_IDH(p, wVal)      EC_AL_SET_VLANHDR_IDH(p, wVal)
#define EC_SET_VLANHDR_IDL(p, wVal)      EC_AL_SET_VLANHDR_IDL(p, wVal)
#define EC_SET_VLANHDR_Prio(p, byVal)    EC_AL_SET_VLANHDR_Prio(p, byVal)
#define EC_SET_VLANHDR_Type(p, wVal)     EC_AL_SET_VLANHDR_Type(p, wVal)
#define EC_SET_VLANHDR_ID(p, wVal)       EC_AL_SET_VLANHDR_ID(p, wVal)
#else

#endif


EC_INLINE(EC_T_WORD EC_VLANHDR_GET_VLANTYPE(PETYPE_VLAN_HEADER p))
{
    return EC_GET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_VLANHDR_OFFS_VLANTYPE)
                     );
}


/*---------------------------------------------------------------------------*/
typedef struct TETHERNET_VLAN_FRAME 
{
    ETHERNET_ADDRESS    Destination;
    ETHERNET_ADDRESS    Source;
    ETYPE_VLAN_HEADER   VLan;
    EC_T_WORD           __FrameType;              /* in host-order */
} EC_PACKED1 ETHERNET_VLAN_FRAME, *PETHERNET_VLAN_FRAME;
#define ETHERNET_VLAN_FRAME_LEN sizeof(ETHERNET_VLAN_FRAME)

#ifdef VLAN_FRAME_SUPPORT
  #define FRAMETYPE_PTR(p)            ((EC_ETHFRM_GET_FRAMETYPE(p)==ETHERNET_FRAME_TYPE_VLAN_SW) ? & ((EC_T_WORD*)p)[8] : &((EC_T_WORD*)p)[6])
#else
  #define FRAMETYPE_PTR(p)            (&((EC_T_WORD*)p)[6])
#endif

/*---------------------------------------------------------------------------*/
#define ETYPE_88A4_TYPE_ECAT        1           /* ECAT header follows */
#define ETYPE_88A4_TYPE_ADS         2           /* ADS header follows */
#define ETYPE_88A4_TYPE_IO          3           /* IO process image follows directly */
#define ETYPE_88A4_TYPE_NV          4           /* Network Variables */
#define ETYPE_88A4_TYPE_MAILBOX     5           /* MAILBOX Header follows */

#define EC_E88A4HDR_OFFS_E88A4FRAMELEN  ((EC_T_BYTE)0)

#define ETYPE_88A4_HEADER_LEN   sizeof(EC_T_WORD)


#ifdef EC_NO_BITFIELDS

typedef EC_T_WORD ETYPE_88A4_HEADER, *PETYPE_88A4_HEADER;

#define EC_88A4HDR_RESET(p)                     ((*(p)) = 0)

#define EC_88A4HDR_GET_E88A4FRAMELEN(pp)        EC_GET_FRM_WORD_BITFIELD(*(pp),0,11)
#define EC_88A4HDR_SET_E88A4FRAMELEN(pp, wVal)  EC_SET_FRM_WORD_BITFIELD(*(pp),wVal,0,11)

#define EC_88A4HDR_GET_E88A4HDRTYPE(pp)         EC_GET_FRM_WORD_BITFIELD(*(pp),12,4)
#define EC_88A4HDR_SET_E88A4HDRTYPE(pp, wVal)   EC_SET_FRM_WORD_BITFIELD(*(pp),wVal,12,4)

#define EC_AL_88A4HDR_RESET                     EC_88A4HDR_RESET            
                                                                            
#define EC_AL_88A4HDR_GET_E88A4FRAMELEN(pp)         EC_88A4HDR_GET_E88A4FRAMELEN((pp))      
#define EC_AL_88A4HDR_SET_E88A4FRAMELEN(pp, wVal)   EC_88A4HDR_SET_E88A4FRAMELEN((pp), (wVal))
                                                                                         
#define EC_AL_88A4HDR_GET_E88A4HDRTYPE(pp)          EC_88A4HDR_GET_E88A4HDRTYPE((pp))       
#define EC_AL_88A4HDR_SET_E88A4HDRTYPE(pp, wVal)    EC_88A4HDR_SET_E88A4HDRTYPE((pp), (wVal)) 

#else /* #ifdef EC_NO_BITFIELDS */

typedef struct TETYPE_88A4_HEADER
{
    union _t_u88a4Hdr
    {
        EC_T_WORD __w88a4Hdr:16;
        struct _t_sw88a4Hdr
        {
            EC_T_WORD   __E88A4FrameLength  : 11;   /*   0 */    /* following bytes */
            EC_T_WORD   Reserved            : 1;    /* 1.3 */        
            EC_T_WORD   __E88A4HdrType      : 4;    /* 1.4 */    /* ETYPE_88A4_TYPE_xxx */
        } EC_PACKED1 sw88a4Hdr;
    } EC_PACKED1 u88a4Hdr;
} EC_PACKED1 ETYPE_88A4_HEADER, *PETYPE_88A4_HEADER;


#define EC_AL_88A4HDR_RESET(p)                      ((p)->u88a4Hdr.__w88a4Hdr) = 0

#define EC_AL_88A4HDR_GET_E88A4FRAMELEN(p)          EC_NTOHS(((p)->u88a4Hdr.sw88a4Hdr.__E88A4FrameLength))
#define EC_AL_88A4HDR_SET_E88A4FRAMELEN(p, wVal)    ((p)->u88a4Hdr.sw88a4Hdr.__E88A4FrameLength) = (wVal)

#define EC_AL_88A4HDR_GET_E88A4HDRTYPE(p)           ((p)->u88a4Hdr.sw88a4Hdr.__E88A4HdrType)
#define EC_AL_88A4HDR_SET_E88A4HDRTYPE(p, wVal)     ((p)->u88a4Hdr.sw88a4Hdr.__E88A4HdrType) = (wVal)

#if !(defined WITHALIGNMENT)

#define EC_88A4HDR_RESET(p)                         EC_AL_88A4HDR_RESET(p)

#define EC_88A4HDR_GET_E88A4FRAMELEN(p)             EC_AL_88A4HDR_GET_E88A4FRAMELEN(p)
#define EC_88A4HDR_SET_E88A4FRAMELEN(p, wVal)       EC_AL_88A4HDR_SET_E88A4FRAMELEN(p, wVal)

#define EC_88A4HDR_GET_E88A4HDRTYPE(p)              EC_AL_88A4HDR_GET_E88A4HDRTYPE(p)
#define EC_88A4HDR_SET_E88A4HDRTYPE(p, wVal)        EC_AL_88A4HDR_SET_E88A4HDRTYPE(p, wVal)

#else  /*!(defined WITHALIGNMENT)*/

#define EC_88A4HDR_RESET(p)                         EC_SETWORD((p), 0)
#define EC_88A4HDR_GET_E88A4HDRTYPE(p)              ((EC_GETWORD((p))>>12)&0xf)
#define EC_88A4HDR_SET_E88A4HDRTYPE(p, wVal)        EC_SETWORD((p), (EC_GETWORD((p))&(~(0xf<<12))) | ((((wVal)&0xf)<<12)))

#define EC_88A4HDR_GET_E88A4FRAMELEN(p)             (EC_GETWORD((p))&0x7ff)
#define EC_88A4HDR_SET_E88A4FRAMELEN(p, wVal)       EC_SETWORD((p), (((wVal)&0x7ff) | (ETYPE_88A4_TYPE_ECAT<<12)))

#endif /*!(defined WITHALIGNMENT)*/

#endif /* #else EC_NO_BITFIELDS */


/*---------------------------------------------------------------------------*/
typedef struct TETHERNET_88A4_FRAME
{
    ETHERNET_FRAME      EthernetFrame;
    ETYPE_88A4_HEADER   E88A4Header;
} EC_PACKED1 ETHERNET_88A4_FRAME, *PETHERNET_88A4_FRAME;
#define ETHERNET_88A4_FRAME_LEN sizeof(ETHERNET_88A4_FRAME)

#define SIZEOF_88A4_FRAME(p)    (ETHERNET_88A4_FRAME_LEN+EC_88A4HDR_GET_E88A4FRAMELEN(p))
#define ENDOF_88A4_FRAME(p)     ((PETHERNET_88A4_FRAME)&(((EC_T_BYTE*)(p))[SIZEOF_88A4_FRAME(p)]))


/*---------------------------------------------------------------------------*/
typedef struct TETHERNET_88A4_VLAN_FRAME
{
    ETHERNET_VLAN_FRAME EthernetFrame;
    ETYPE_88A4_HEADER   E88A4Header;
} EC_PACKED1 ETHERNET_88A4_VLAN_FRAME, *PETHERNET_88A4_VLAN_FRAME;
#define ETHERNET_88A4_VLAN_FRAME_LEN    sizeof(ETHERNET_88A4_VLAN_FRAME)

#define SIZEOF_88A4_VLAN_FRAME(p)       (ETHERNET_88A4_VLAN_FRAME_LEN+EC_88A4HDR_GET_E88A4FRAMELEN(p))
#define ENDOF_88A4_VLANFRAME(p)         ((PETHERNET_88A4_VLAN_FRAME)&(((EC_T_BYTE*)(p))[SIZEOF_88A4_VLAN_FRAME(p)]))

/*---------------------------------------------------------------------------*/


#define EC_HEADER_IDX_EXTERN_VALUE  0xFF

#define EC_CMDHDR_OFFS_CMDIDX       ((EC_T_BYTE)0)
#define EC_CMDHDR_OFFS_CMDIDX_CMD   ((EC_T_BYTE)0)
#define EC_CMDHDR_OFFS_CMDIDX_IDX   ((EC_T_BYTE)1)

#define EC_CMDHDR_OFFS_ADDR         ((EC_T_BYTE)2)
#define EC_CMDHDR_OFFS_ADDR_ADP     ((EC_T_BYTE)2)
#define EC_CMDHDR_OFFS_ADDR_ADO     ((EC_T_BYTE)4)

#define EC_CMDHDR_OFFS_LEN          ((EC_T_BYTE)6)
#define EC_CMDHDR_OFFS_LEN_LEN      ((EC_T_BYTE)6)
/*#define EC_CMDHDR_OFFS_LEN_NEXT     ((EC_T_BYTE)6)*/
#define EC_CMDHDR_OFFS_LEN_NEXT_BYTE ((EC_T_BYTE)7)


#define EC_CMDHDR_OFFS_IRQ          ((EC_T_BYTE)8)

                    

/* General EtherCAT telegram header */
typedef struct TETYPE_EC_CMD_HEADER
{
    union _t_uCmdIdx
    {
        EC_T_WORD __wCmdIdx:16;     /* 0 */
        struct _t_swCmdIdx
        {
            EC_T_BYTE   byCmd;      /* 0 */
            EC_T_BYTE   byIdx;      /* 1 */
        } EC_PACKED1 swCmdIdx;
    } EC_PACKED1 uCmdIdx;
    union _t_uAddr
    {
        struct _t_sladdr
        {
            EC_T_WORD   __adp;      /* 2 */
            EC_T_WORD   __ado;      /* 4 */
        } EC_PACKED1 sladdr;
        EC_T_DWORD __laddr;         /* 2 */
    } EC_PACKED1 uAddr;
    union _t_uLen
    {
        struct _t_slength
        {
#ifndef EC_NO_BITFIELDS
            EC_T_WORD   __len : 11; /* 6.0 */
            EC_T_WORD   res : 4;    /* 6.12 */
            EC_T_WORD   __bNext: 1; /* 6.15 */
#endif
        } EC_PACKED1 slength;
        EC_T_WORD __length;         /* 6 */
    } EC_PACKED1 uLen;
    EC_T_WORD __irq;                /* 8 */
} EC_PACKED1 ETYPE_EC_CMD_HEADER, *PETYPE_EC_CMD_HEADER;
#define ETYPE_EC_CMD_HEADER_LEN     sizeof(ETYPE_EC_CMD_HEADER)
#define ETYPE_EC_WKC_LEN            sizeof(EC_T_WORD)
#define ETYPE_EC_OVERHEAD           (ETYPE_EC_CMD_HEADER_LEN+ETYPE_EC_WKC_LEN)

/* access macros */
/* Offset 0 */
#ifdef EC_BIG_ENDIAN
#define EC_AL_ICMDHDR_GET_CMDIDX(p)             EC_WORDSWAP((p)->uCmdIdx.__wCmdIdx)
#else
#define EC_AL_ICMDHDR_GET_CMDIDX(p)             ((p)->uCmdIdx.__wCmdIdx)
#endif
#define EC_AL_ICMDHDR_GET_CMDIDX_CMD(p)         ((p)->uCmdIdx.swCmdIdx.byCmd)
#define EC_AL_ICMDHDR_GET_CMDIDX_IDX(p)         ((p)->uCmdIdx.swCmdIdx.byIdx)
                                                
/* Offset 2 */
#ifdef EC_BIG_ENDIAN
#define EC_AL_ICMDHDR_GET_ADDR(p)               EC_DWORDSWAP((p)->uAddr.__laddr)
#define EC_AL_ICMDHDR_GET_ADDR_ADO(p)           EC_WORDSWAP((p)->uAddr.sladdr.__ado)
#define EC_AL_ICMDHDR_GET_ADDR_ADP(p)           EC_WORDSWAP((p)->uAddr.sladdr.__adp)

#define EC_AL_ICMDHDR_SET_ADDR(p, dwVal)        ((p)->uAddr.__laddr) = EC_DWORDSWAP(dwVal)
#define EC_AL_ICMDHDR_SET_ADDR_ADO(p, wVal)     ((p)->uAddr.sladdr.__ado) = EC_WORDSWAP(wVal)
#define EC_AL_ICMDHDR_SET_ADDR_ADP(p, wVal)     ((p)->uAddr.sladdr.__adp) = EC_WORDSWAP(wVal)
#else
#define EC_AL_ICMDHDR_GET_ADDR(p)               ((p)->uAddr.__laddr)
#define EC_AL_ICMDHDR_GET_ADDR_ADO(p)           ((p)->uAddr.sladdr.__ado)
#define EC_AL_ICMDHDR_GET_ADDR_ADP(p)           ((p)->uAddr.sladdr.__adp)

#define EC_AL_ICMDHDR_SET_ADDR(p, dwVal)        ((p)->uAddr.__laddr) = (dwVal)
#define EC_AL_ICMDHDR_SET_ADDR_ADO(p, wVal)     ((p)->uAddr.sladdr.__ado) = (wVal)
#define EC_AL_ICMDHDR_SET_ADDR_ADP(p, wVal)     ((p)->uAddr.sladdr.__adp) = (wVal)
#endif
       
/* Offset 6 */
#ifdef EC_BIG_ENDIAN
#define EC_AL_ICMDHDR_GET_LEN(p)                EC_DWORDSWAP((p)->uLen.__length)
#else
#define EC_AL_ICMDHDR_GET_LEN(p)                ((p)->uLen.__length)
#endif
#if (!defined EC_NO_BITFIELDS)
#define EC_AL_ICMDHDR_GET_LEN_LEN(p)            ((p)->uLen.slength.__len)
#define EC_AL_CMDHDRLEN_GET_NEXT(puLen)         ((puLen)->slength.__bNext)
#define EC_AL_CMDHDRLEN_GET_LEN(puLen)          ((puLen)->slength.__len)
#define EC_AL_CMDHDRLEN_SET_LEN(puLen, wVal)    ((puLen)->slength.__len) = wVal
#endif

#ifdef EC_BIG_ENDIAN
#define EC_AL_CMDHDRLEN_SET_LEN_AND_NEXT(puLen, wVal, bNext) \
                                                ((puLen)->__length) = EC_WORDSWAP((EC_T_WORD)(((wVal)&0x7ff) | (((bNext)&1)<<15)))
#else
#define EC_AL_CMDHDRLEN_SET_LEN_AND_NEXT(puLen, wVal, bNext) \
                                                ((puLen)->__length) = ((EC_T_WORD)(((wVal)&0x7ff) | (((bNext)&1)<<15)))
#endif

/* Offset 8 */
#ifdef EC_BIG_ENDIAN
#define EC_AL_ICMDHDR_GET_IRQ(p)                EC_WORDSWAP((p)->__irq)
#else
#define EC_AL_ICMDHDR_GET_IRQ(p)                ((p)->__irq)
#endif

                                                
#if (defined EC_NO_BITFIELDS)

typedef EC_T_WORD TETYPE_EC_CMD_HEADER_LENGTH;

#define EC_CMDHDRLEN_GET_NEXT(puLen)        EC_GET_FRM_WORD_BITFIELD((puLen)->__length,15,1)
#define EC_CMDHDRLEN_SET_NEXT(puLen,bNext)  EC_SET_FRM_WORD_BITFIELD((puLen)->__length,(bNext),15,1)

#define EC_CMDHDRLEN_GET_LEN(puLen)         EC_GET_FRM_WORD_BITFIELD((puLen)->__length,0,11)
#define EC_CMDHDRLEN_SET_LEN(puLen, wVal)   EC_SET_FRM_WORD_BITFIELD((puLen)->__length,wVal,0,11)

#endif /* #ifdef EC_NO_BITFIELDS */

#if (defined EC_BIG_ENDIAN)

#define EC_CMDHDRLEN_SET_LEN_AND_NEXT(puLen, wVal, bNext)   \
    EC_CMDHDRLEN_SET_LEN((puLen), (wVal))                   \
    EC_CMDHDRLEN_SET_NEXT((puLen),(bNext))
#define EC_AL_ICMDHDR_GET_LEN_LEN               EC_ICMDHDR_GET_LEN_LEN
#define EC_AL_CMDHDRLEN_GET_NEXT(puLen)         EC_CMDHDRLEN_GET_NEXT((puLen))
#define EC_AL_CMDHDRLEN_GET_LEN(puLen)          EC_CMDHDRLEN_GET_LEN((puLen))
#define EC_AL_CMDHDRLEN_SET_LEN(puLen, wVal)    EC_CMDHDRLEN_SET_LEN((puLen),(wVal))

/*#define EC_ICMDHDR_GET_CMDIDX(p)            EC_AL_ICMDHDR_GET_CMDIDX(p)*/
EC_INLINE (EC_T_WORD EC_ICMDHDR_GET_CMDIDX(PETYPE_EC_CMD_HEADER p))
{
    return EC_GET_FRM_WORD(
        (((EC_T_BYTE*)p)+EC_CMDHDR_OFFS_CMDIDX)
        );
}
#define EC_ICMDHDR_GET_CMDIDX_CMD(p)        EC_AL_ICMDHDR_GET_CMDIDX_CMD(p)
#define EC_ICMDHDR_GET_CMDIDX_IDX(p)        EC_AL_ICMDHDR_GET_CMDIDX_IDX(p)

EC_INLINE (EC_T_DWORD EC_ICMDHDR_GET_ADDR(PETYPE_EC_CMD_HEADER p))
{
    return EC_GET_FRM_DWORD(
        (((EC_T_BYTE*)p)+EC_CMDHDR_OFFS_ADDR)
        );
}
EC_INLINE (EC_T_WORD EC_ICMDHDR_GET_ADDR_ADO(PETYPE_EC_CMD_HEADER p))
{
    return EC_GET_FRM_WORD(
        (((EC_T_BYTE*)p)+EC_CMDHDR_OFFS_ADDR_ADO)
        );
}
EC_INLINE (EC_T_WORD EC_ICMDHDR_GET_ADDR_ADP(PETYPE_EC_CMD_HEADER p))
{
    return EC_GET_FRM_WORD(
        (((EC_T_BYTE*)p)+EC_CMDHDR_OFFS_ADDR_ADP)
        );
}
EC_INLINE (EC_T_WORD EC_ICMDHDR_GET_LEN(PETYPE_EC_CMD_HEADER p))
{
    return EC_GET_FRM_WORD(
        (((EC_T_BYTE*)p)+EC_CMDHDR_OFFS_LEN)
        );
}
EC_INLINE (EC_T_WORD EC_ICMDHDR_GET_LEN_LEN(PETYPE_EC_CMD_HEADER p))
{
    return (EC_T_WORD)(
        EC_GET_FRM_WORD(
        (((EC_T_BYTE*)p)+EC_CMDHDR_OFFS_LEN_LEN)
        )&((1<<11)-1));
}
#ifdef EC_BIG_ENDIAN
#define EC_ICMDHDR_GET_IRQ(p)   SWAPWORD(((p)->__irq))
#else
#define EC_ICMDHDR_GET_IRQ(p)   ((p)->__irq)
#endif

EC_INLINE (EC_T_VOID EC_ICMDHDR_SET_ADDR(PETYPE_EC_CMD_HEADER p, EC_T_DWORD dwVal))
{
    EC_SET_FRM_DWORD(
        (((EC_T_BYTE*)p)+EC_CMDHDR_OFFS_ADDR),
        dwVal
        );
}
EC_INLINE (EC_T_VOID EC_ICMDHDR_SET_ADDR_ADO(PETYPE_EC_CMD_HEADER p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD(
        (((EC_T_BYTE*)p)+EC_CMDHDR_OFFS_ADDR_ADO), 
        wVal
        );
}
EC_INLINE (EC_T_VOID EC_ICMDHDR_SET_ADDR_ADP(PETYPE_EC_CMD_HEADER p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD(
        (((EC_T_BYTE*)p)+EC_CMDHDR_OFFS_ADDR_ADP), 
        wVal
        );
}
EC_INLINE (EC_T_VOID EC_ICMDHDR_SET_IRQ(PETYPE_EC_CMD_HEADER p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD(
        (((EC_T_BYTE*)p)+EC_CMDHDR_OFFS_IRQ), 
        wVal
        );
}

#elif (defined WITHALIGNMENT)
  
/* Offset 0 */
#define EC_ICMDHDR_GET_CMDIDX(p)            (EC_GETWORD((p)))
#define EC_ICMDHDR_GET_CMDIDX_CMD(p)        (((EC_T_BYTE*)(p))[0])
#define EC_ICMDHDR_GET_CMDIDX_IDX(p)        (((EC_T_BYTE*)(p))[1])

/* Offset 2 */
#define EC_ICMDHDR_GET_ADDR(p)              EC_GETDWORD( (((EC_T_BYTE*)(p))+ EC_CMDHDR_OFFS_ADDR) )
#define EC_ICMDHDR_GET_ADDR_ADO(p)          EC_GETWORD( (((EC_T_BYTE*)(p))+ EC_CMDHDR_OFFS_ADDR_ADO) )
#define EC_ICMDHDR_GET_ADDR_ADP(p)          EC_GETWORD( (((EC_T_BYTE*)(p))+ EC_CMDHDR_OFFS_ADDR_ADP) )

#define EC_ICMDHDR_SET_ADDR(p, dwVal)       EC_SETDWORD( (((EC_T_BYTE*)(p))+ EC_CMDHDR_OFFS_ADDR), (dwVal))
#define EC_ICMDHDR_SET_ADDR_ADO(p, wVal)    EC_SETWORD( (((EC_T_BYTE*)(p))+ EC_CMDHDR_OFFS_ADDR_ADO), (wVal))
#define EC_ICMDHDR_SET_ADDR_ADP(p, wVal)    EC_SETWORD( (((EC_T_BYTE*)(p))+ EC_CMDHDR_OFFS_ADDR_ADP), (wVal))
                                            
/* Offset 6 */
#define EC_ICMDHDR_GET_LEN(p)               ((EC_T_WORD)(EC_GETWORD( (((EC_T_BYTE*)(p))+EC_CMDHDR_OFFS_LEN_LEN))))
#define EC_ICMDHDR_GET_LEN_LEN(p)           ((EC_T_WORD)(EC_GETWORD( (((EC_T_BYTE*)(p)))+EC_CMDHDR_OFFS_LEN_LEN )&0x7ff))

#define EC_CMDHDRLEN_GET_NEXT(puLen)        ((EC_T_WORD)((EC_GETWORD((((EC_T_BYTE*)(puLen))))>>15)&0x1))

#define EC_CMDHDRLEN_GET_LEN(puLen)         ((EC_T_WORD)((EC_GETWORD((((EC_T_BYTE*)(puLen)))))&0x7ff))

#define EC_CMDHDRLEN_SET_LEN_AND_NEXT(puLen, wVal, bNext)   EC_SETWORD((puLen), ((EC_T_WORD)(((wVal)&0x7ff) | (((bNext)&1)<<15))) )


/* Offset 8 */
#define EC_ICMDHDR_GET_IRQ(p)               EC_GETWORD((((EC_T_BYTE*)(p))+EC_CMDHDR_OFFS_IRQ))

#else /* defined WITHALIGMENT */

/* Offset 0 */
#define EC_ICMDHDR_GET_CMDIDX(p)            EC_AL_ICMDHDR_GET_CMDIDX(p)
#define EC_ICMDHDR_GET_CMDIDX_CMD(p)        EC_AL_ICMDHDR_GET_CMDIDX_CMD(p)
#define EC_ICMDHDR_GET_CMDIDX_IDX(p)        EC_AL_ICMDHDR_GET_CMDIDX_IDX(p)
   
/* Offset 2 */
#define EC_ICMDHDR_GET_ADDR(p)              EC_AL_ICMDHDR_GET_ADDR(p)
#define EC_ICMDHDR_GET_ADDR_ADO(p)          EC_AL_ICMDHDR_GET_ADDR_ADO(p)
#define EC_ICMDHDR_GET_ADDR_ADP(p)          EC_AL_ICMDHDR_GET_ADDR_ADP(p)

#define EC_ICMDHDR_SET_ADDR(p, dwVal)       EC_AL_ICMDHDR_SET_ADDR(p, dwVal)
#define EC_ICMDHDR_SET_ADDR_ADO(p, wVal)    EC_AL_ICMDHDR_SET_ADDR_ADO(p, wVal)
#define EC_ICMDHDR_SET_ADDR_ADP(p, wVal)    EC_AL_ICMDHDR_SET_ADDR_ADP(p, wVal)
                                         
/* Offset 6 */
#define EC_ICMDHDR_GET_LEN(p)               EC_AL_ICMDHDR_GET_LEN(p)
#define EC_ICMDHDR_GET_LEN_LEN(p)           EC_AL_ICMDHDR_GET_LEN_LEN(p)

#define EC_CMDHDRLEN_GET_NEXT(puLen)        EC_AL_CMDHDRLEN_GET_NEXT(puLen)

#define EC_CMDHDRLEN_GET_LEN(puLen)         EC_AL_CMDHDRLEN_GET_LEN(puLen)
#define EC_CMDHDRLEN_SET_LEN(puLen, wVal)   EC_AL_CMDHDRLEN_SET_LEN(puLen, wVal)

#define EC_CMDHDRLEN_SET_LEN_AND_NEXT(puLen, wVal, bNext)   EC_AL_CMDHDRLEN_SET_LEN_AND_NEXT(puLen, wVal, bNext)

/* Offset 8 */
#define EC_ICMDHDR_GET_IRQ(p)               EC_AL_ICMDHDR_GET_IRQ(p)

#endif /* #else EC_NO_BITFIELDS */



/* EC_INLINE (EC_T_VOID EC_ICMDHDR_SET_CMDIDX(PETYPE_EC_CMD_HEADER p, EC_T_WORD wVal))
{
    EC_SETWORD(
        (((EC_T_BYTE*)p)+EC_CMDHDR_OFFS_CMDIDX), 
        wVal
              );
} */
EC_INLINE (EC_T_VOID EC_ICMDHDR_SET_LEN(PETYPE_EC_CMD_HEADER p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD(
        (((EC_T_BYTE*)p)+EC_CMDHDR_OFFS_LEN), 
        wVal
              );
}

EC_INLINE (EC_T_VOID EC_ICMDHDR_SET_LEN_LEN(PETYPE_EC_CMD_HEADER p, EC_T_WORD wVal))
{
EC_T_WORD wOldVal;
EC_T_WORD wMask;
    wOldVal = EC_ICMDHDR_GET_LEN((p));          /* get old value (as the len is shared with other bits) */
    wMask = (EC_T_WORD)(~((1<<11)-1));
    wOldVal &= wMask;                           /* eliminate old len value (only relevant bits) */
    wVal = (EC_T_WORD)(wVal & ((1<<11)-1));     /* mask irrelevant bits */
    wVal = (EC_T_WORD)(wOldVal | wVal);         /* determine new length together with other bits untouched */
    EC_ICMDHDR_SET_LEN((p),wVal);               /* store it */
}
EC_INLINE (EC_T_VOID EC_ICMDHDR_SET_LEN_NEXT(PETYPE_EC_CMD_HEADER p, EC_T_BOOL bVal))
{
    if(bVal)
    {
        (((EC_T_BYTE*)p)+EC_CMDHDR_OFFS_LEN_NEXT_BYTE)[0] |=((EC_T_BYTE)0x80);
    }
    else
    {
        (((EC_T_BYTE*)p)+EC_CMDHDR_OFFS_LEN_NEXT_BYTE)[0] &=((EC_T_BYTE)~0x80);
    }

/*
    
    EC_SET_FRM_WORD(
        (((EC_T_BYTE*)p)+EC_CMDHDR_OFFS_LEN), 
        ((EC_T_WORD)((EC_ICMDHDR_GET_LEN(p)&(~(1<<15)))|((bVal&1)<<15)))
              );
*/
}

EC_INLINE (EC_T_WORD ETYPE_EC_CMD_GETLEN(PETYPE_EC_CMD_HEADER p))
{
    return (EC_T_WORD)(ETYPE_EC_OVERHEAD + EC_ICMDHDR_GET_LEN_LEN(p));
}
EC_INLINE (EC_T_VOID ETYPE_EC_CMD_SETWKC(PETYPE_EC_CMD_HEADER p, EC_T_WORD wVal))
{
EC_T_BYTE* pbyCur = EC_NULL;

    pbyCur = &(((EC_T_BYTE*)p)[ETYPE_EC_CMD_HEADER_LEN + EC_ICMDHDR_GET_LEN_LEN(p)]);

    EC_SET_FRM_WORD(pbyCur, wVal);
}
EC_INLINE (EC_T_WORD ETYPE_EC_CMD_GETWKC(PETYPE_EC_CMD_HEADER p))
{
EC_T_BYTE* pbyCur = EC_NULL;

    /*pbyCur = &(((EC_T_BYTE*)p)[ETYPE_EC_CMD_HEADER_LEN + p->uLen.slength.__len]);*/
    pbyCur = &(((EC_T_BYTE*)p)[ETYPE_EC_CMD_HEADER_LEN + EC_ICMDHDR_GET_LEN_LEN( p )]);

    return EC_GET_FRM_WORD(pbyCur);
}
EC_INLINE (EC_T_WORD ETYPE_EC_CMD_GETWKCOFF(PETYPE_EC_CMD_HEADER p))
{
    return (EC_T_WORD)(ETYPE_EC_CMD_HEADER_LEN + EC_ICMDHDR_GET_LEN_LEN(p));
}
EC_INLINE (PETYPE_EC_CMD_HEADER NEXT_EcCmdHeader(PETYPE_EC_CMD_HEADER p))
{
PETYPE_EC_CMD_HEADER pRetVal = 0;

    pRetVal = (PETYPE_EC_CMD_HEADER)(&((EC_T_BYTE*)p)[EC_ICMDHDR_GET_LEN_LEN(p) + ETYPE_EC_OVERHEAD]);

    return pRetVal;
}

/* FMMU configuration command */
typedef struct _ETYPE_EC_T_FMMU_CFG_CMD
{
    EC_T_DWORD  dwLogStartAddr;     /* 0x00 */
    EC_T_WORD   wLength;            /* 0x04 */
    EC_T_BYTE   byStartBit;         /* 0x06 */
    EC_T_BYTE   byStopBit;          /* 0x07 */
    EC_T_WORD   wPhysStart;         /* 0x08 */
    EC_T_BYTE   byPhysStartBit;     /* 0x0A */
    EC_T_BYTE   byType;             /* 0x0B */
    EC_T_BYTE   byActivate;         /* 0x0C */
    EC_T_BYTE   byReserved;         /* 0x0D */
    EC_T_WORD   wReserved;          /* 0x0E */
} EC_PACKED1 ETYPE_EC_T_FMMU_CFG_CMD, *PETYPE_EC_T_FMMU_CFG_CMD;

#define FMMU_CFG_CMD_TYPE_READ  ((EC_T_BYTE)0x01)
#define FMMU_CFG_CMD_TYPE_WRITE ((EC_T_BYTE)0x02)

#define ETYPE_EC_T_FMMU_CFG_CMD_SIZE sizeof(ETYPE_EC_T_FMMU_CFG_CMD)

#define MAX_EC_CMD_PER_FRAME  124   /* ((ETHERNET_MAX_FRAME_LEN - 0x10)/ETYPE_EC_OVERHEAD) */

#define MAX_EC_DATA_LEN (ETHERNET_MAX_FRAME_LEN - ETHERNET_88A4_FRAME_LEN - ETYPE_EC_OVERHEAD)

/* consistency checks */
#if MAX_EC_CMD_PER_FRAME > (255 - EC_HEAD_IDX_SLAVECMD)
#error MAX_EC_CMD_PER_FRAME > (255 - EC_HEAD_IDX_SLAVECMD)
#endif

/*---------------------------------------------------------------------------*/
typedef struct TETHERNET_88A4_HEADER
{
    ETYPE_88A4_HEADER E88A4Header;
    struct _t_sETECAT_88a4Hdr
    {   /* ETYPE_88A4_TYPE_ECAT */
        ETYPE_EC_CMD_HEADER FirstEcCmdHeader;
    } EC_PACKED1
    sETECAT;
} EC_PACKED1 ETHERNET_88A4_HEADER, *PETHERNET_88A4_HEADER;

/*---------------------------------------------------------------------------*/
typedef struct TETHERNET_88A4_MAX_FRAME
{
    ETHERNET_FRAME    EthernetFrame;
    ETYPE_88A4_HEADER E88A4Header;
    struct _t_sETECAT_maxFrame
    {   /* ETYPE_88A4_TYPE_ECAT */
        ETYPE_EC_CMD_HEADER FirstEcCmdHeader;
    } EC_PACKED1
    sETECAT;
    EC_T_BYTE   abyData[MAX_EC_DATA_LEN];
    EC_T_WORD   __wWkc;
} EC_PACKED1 ETHERNET_88A4_MAX_FRAME, *PETHERNET_88A4_MAX_FRAME;

typedef struct TETHERNET_88A4_MAX_VLAN_FRAME
{
    ETHERNET_VLAN_FRAME    EthernetFrame;
    ETYPE_88A4_HEADER E88A4Header;
    struct _t_sETECAT_maxVlanFrame
    {   /* ETYPE_88A4_TYPE_ECAT */
        ETYPE_EC_CMD_HEADER FirstEcCmdHeader;
    } EC_PACKED1
        sETECAT;
    EC_T_BYTE   abyData[MAX_EC_DATA_LEN-3];
    EC_T_WORD   __wWkc;
} EC_PACKED1 ETHERNET_88A4_MAX_VLAN_FRAME, *PETHERNET_88A4_MAX_VLAN_FRAME;

/*---------------------------------------------------------------------------*/

typedef struct TEcCmdDesc
{
    ETYPE_EC_CMD_HEADER     EcCmdHeader;                /* command header 10 Bytes     = 10*/
    EC_T_WORD               reserved1;                  /* ETYPE_EC_CMD_HEADER_LEN + 0 = 10 */
    EC_T_BOOL               bChkCntRecv;                /* ETYPE_EC_CMD_HEADER_LEN + 2 = 12 */
    EC_T_BOOL               bForceNewFrame;             /* ETYPE_EC_CMD_HEADER_LEN + 6 = 16 */

    EC_T_WORD               wWkcSend;                   /* ETYPE_EC_CMD_HEADER_LEN + 10 = 20 */
    EC_T_WORD               cmdSize;                    /* ETYPE_EC_CMD_HEADER_LEN + 12 = 22 */
    EC_T_WORD               imageOffs[2];               /* ETYPE_EC_CMD_HEADER_LEN + 14 = 24 */
    EC_T_WORD               imageSize[2];               /* ETYPE_EC_CMD_HEADER_LEN + 18 = 28 */
#define EC_ECCMDDESC_PARMCOPYINPUTS         0x0001
#define EC_ECCMDDESC_PARMCOPYOUTPUTS        0x0002
#define EC_ECCMDDESC_PARMLOGMBOXSTATE       0x0004
#define EC_ECCMDDESC_PARMDCDEPLOYMENT       0x0008
    EC_T_WORD               cpParm;
    EC_T_WORD               cntRecv;                    /* ETYPE_EC_CMD_HEADER_LEN + 24 = 34 */
    EC_T_BYTE               byConfOpStatesMask;         /* ETYPE_EC_CMD_HEADER_LEN + 26 = 36 */
                                                        /* configuration operational state (../config/cyclic/frame/cmd/State) */
    EC_T_BYTE               reserved2[3];               /* ETYPE_EC_CMD_HEADER_LEN + 21 = 37 */
} EC_PACKED1 EcCmdDesc, *PEcCmdDesc;

#define ECCMDDESC_LEN    sizeof(EcCmdDesc)


/*---------------------------------------------------------------------------*/
/* Slave-to-Slave copy information                                           */
/*---------------------------------------------------------------------------*/
/* Copy information for slave to slave communication. The master has to copy valid 
input data of this command from the source offest (bit offs in the complete process image) 
to a destination offset. */
#define     MAX_NUMOF_COPY_ENTRYS       64
typedef struct
{
    EC_T_WORD  wSrcBitOffs; 
    EC_T_WORD  wDstBitOffs;
    EC_T_WORD  wBitSize;
    EC_T_WORD  wTaskId;
} EC_PACKED1 EC_T_CYC_COPY_INFO;

/*---------------------------------------------------------------------------*/
typedef struct TEcCycCmdConfig
{
    ETHERNET_ADDRESS        macTarget;                                      /* 0 */
    EC_T_WORD               wRes;                                           /* ETHERNET_ADDRESS_LEN */
    EC_T_WORD               wSize;                                          /* ETHERNET_ADDRESS_LEN + 2 */
    EC_T_WORD               awNumCycCmdsPerState[EC_NUM_CYCCMD_STATES];   /* ETHERNET_ADDRESS_LEN + 4 */
                                                                            /* number of cyclic ecat telegram to be sent in the corresponding op state */
    EC_T_WORD               syncMapId;                                    /* ETHERNET_ADDRESS_LEN + 4 + 2*EC_NUM_CYCCMD_STATES + 0 */
    EC_T_WORD               cdlNo;                                        /* ETHERNET_ADDRESS_LEN + 4 + 2*EC_NUM_CYCCMD_STATES + 2 */
                                                                            /*  0, 1, 2, 3...   0 = highest prio */
    ETYPE_VLAN_HEADER       vlanHeader;                                     /* ETHERNET_ADDRESS_LEN + 4 + 2*EC_NUM_CYCCMD_STATES + 4 + 0 */
    EC_T_BYTE               ecatMaster;                                     /* ETHERNET_ADDRESS_LEN + 4 + 2*EC_NUM_CYCCMD_STATES + 4 + ETYPE_VLAN_HEADER_LEN + 0 */
    EC_T_BYTE               byAllOpStatesMask;                              /* ETHERNET_ADDRESS_LEN + 4 + 2*EC_NUM_CYCCMD_STATES + 4 + ETYPE_VLAN_HEADER_LEN + 1 */
    EC_T_WORD               wNumOfCopyInfos;                                /* number of slave-to-slave copy info entrys */
    EC_T_CYC_COPY_INFO      aCopyInfo[MAX_NUMOF_COPY_ENTRYS];               /* copy info */
} EcCycCmdConfig, *PEcCycCmdConfig;


/*---------------------------------------------------------------------------*/
#define ECAT_INITCMD_I_P            0x0001  
#define ECAT_INITCMD_P_S            0x0002
#define ECAT_INITCMD_P_I            0x0004
#define ECAT_INITCMD_S_P            0x0008  
#define ECAT_INITCMD_S_O            0x0010
#define ECAT_INITCMD_S_I            0x0020  
#define ECAT_INITCMD_O_S            0x0040
#define ECAT_INITCMD_O_P            0x0080  
#define ECAT_INITCMD_O_I            0x0100  
#define ECAT_INITCMD_I_B            0x0200
#define ECAT_INITCMD_B_I            0x0400

#define ECAT_INITCMD_OSP_I          0x0124
#define ECAT_INITCMD_OSP_I__I_P     0x0125
#define ECAT_INITCMD_SP_SI_OP_OI    0x01A8
#define ECAT_INITCMD_IP_SP_SI_OP_OI 0x01A9
#define ECAT_INITCMD_I_PB           0x0201
#define ECAT_INITCMD_BI_IP          0x0401
#define ECAT_INITCMD_IP_PS          0x0003
#define ECAT_INITCMD_BACKTO_I       0x0524
#define ECAT_INITCMD_BACKTO_P       0x0088
#define ECAT_INITCMD_BEFORE         0x8000
#define ECAT_INITCMD_OSP_I__I_BP    0x0325

#define ECAT_INITCMD_ADR_ERROR      0x7FFE
#define ECAT_INITCMD_FAILURE        0x7FFF

#define EC_ECINITCMDDESC_OFFS_TRANSITION        ((EC_T_BYTE)ETYPE_EC_CMD_HEADER_LEN)
#define EC_ECINITCMDDESC_OFFS_CNT               ((EC_T_BYTE)ETYPE_EC_CMD_HEADER_LEN + 2)
#define EC_ECINITCMDDESC_OFFS_CMTLEN            ((EC_T_BYTE)ETYPE_EC_CMD_HEADER_LEN + 4)
#define EC_ECINITCMDDESC_OFFS_INITCMDTIMEOUT    ((EC_T_BYTE)ETYPE_EC_CMD_HEADER_LEN + 7)
#define EC_ECINITCMDDESC_OFFS_RETRIES           ((EC_T_BYTE)ETYPE_EC_CMD_HEADER_LEN + 9)

typedef struct TEcInitCmdDesc
{
    ETYPE_EC_CMD_HEADER         EcCmdHeader;                /* 0 */
    EC_T_WORD                   __transition;               /* ETYPE_EC_CMD_HEADER_LEN + 0 = 10 */

    EC_T_WORD                   __cnt;                      /* ETYPE_EC_CMD_HEADER_LEN + 2 */
    EC_T_WORD                   __cmtLen;                   /* ETYPE_EC_CMD_HEADER_LEN + 4 */
                                                            /* (excl. \0) */
    EC_T_BYTE                   byFlags;                    /* ETYPE_EC_CMD_HEADER_LEN + 6 */
/*
    EC_T_BYTE                   __newCycle          : 1;    / * ETYPE_EC_CMD_HEADER_LEN + 6.0 * /
    
    EC_T_BYTE                   __newFrame          : 1;    / * ETYPE_EC_CMD_HEADER_LEN + 6.1 * /
    EC_T_BYTE                   __validate          : 1;    / * ETYPE_EC_CMD_HEADER_LEN + 6.2 * /
    
    EC_T_BYTE                   __validateMask      : 1;    / * ETYPE_EC_CMD_HEADER_LEN + 6.3 * /
    EC_T_BYTE                   reserved            : 4;    / * ETYPE_EC_CMD_HEADER_LEN + 6.4 * /
*/

    EC_T_WORD                   __wInitCmdTimeout;          /* ETYPE_EC_CMD_HEADER_LEN + 7 */
                                                            /* in ms */
    EC_T_WORD                   __retries;                  /* ETYPE_EC_CMD_HEADER_LEN + 9 */
    EC_T_BYTE                   reserved2;                  /* ETYPE_EC_CMD_HEADER_LEN + 11 */
} EC_PACKED1 EcInitCmdDesc, *PEcInitCmdDesc;

EC_INLINE(EC_T_WORD EC_ECINITCMDDESC_GET_TRANSITION(PEcInitCmdDesc p))
{
    return EC_GETWORD(
        (((EC_T_PBYTE)p)+EC_ECINITCMDDESC_OFFS_TRANSITION)
                     );
}

EC_INLINE(EC_T_WORD EC_ECINITCMDDESC_GET_CNT(PEcInitCmdDesc p))
{
    return EC_GETWORD(
        (((EC_T_PBYTE)p)+EC_ECINITCMDDESC_OFFS_CNT)
                     );
}

EC_INLINE(EC_T_WORD EC_ECINITCMDDESC_GET_CMTLEN(PEcInitCmdDesc p))
{
    return EC_GETWORD(
        (((EC_T_PBYTE)p)+EC_ECINITCMDDESC_OFFS_CMTLEN)
                     );
}

EC_INLINE(EC_T_BOOL EC_ECINITCMDDESC_GET_VALIDATE(PEcInitCmdDesc p))
{
    return (p->byFlags & (1<<2));
}

EC_INLINE(EC_T_BOOL EC_ECINITCMDDESC_GET_VALIDATEMASK(PEcInitCmdDesc p))
{
    return (p->byFlags & (1<<3));
}

EC_INLINE(EC_T_WORD EC_ECINITCMDDESC_GET_INITCMDTIMEOUT(PEcInitCmdDesc p))
{
    return EC_GETWORD(
        (((EC_T_PBYTE)p)+EC_ECINITCMDDESC_OFFS_INITCMDTIMEOUT)
                     );
}

EC_INLINE(EC_T_WORD EC_ECINITCMDDESC_GET_RETRIES(PEcInitCmdDesc p))
{
    return EC_GETWORD(
        (((EC_T_PBYTE)p)+EC_ECINITCMDDESC_OFFS_RETRIES)
                     );
}

EC_INLINE(EC_T_VOID EC_ECINITCMDDESC_SET_INITCMDTIMEOUT(PEcInitCmdDesc p, EC_T_WORD wVal))
{
    EC_SETWORD(
        (((EC_T_PBYTE)p)+EC_ECINITCMDDESC_OFFS_INITCMDTIMEOUT),
        wVal
              );
}

EC_INLINE(EC_T_VOID EC_ECINITCMDDESC_SET_CMTLEN(PEcInitCmdDesc p, EC_T_WORD wVal))
{
    EC_SETWORD(
        (((EC_T_PBYTE)p)+EC_ECINITCMDDESC_OFFS_CMTLEN),
        wVal
              );
}

EC_INLINE(EC_T_VOID EC_ECINITCMDDESC_SET_TRANSITION(PEcInitCmdDesc p, EC_T_WORD wVal))
{
    EC_SETWORD(
        (((EC_T_PBYTE)p)+EC_ECINITCMDDESC_OFFS_TRANSITION),
        wVal
              );
}

EC_INLINE(EC_T_VOID EC_ECINITCMDDESC_SET_RETRIES(PEcInitCmdDesc p, EC_T_WORD wVal))
{
    EC_SETWORD(
        (((EC_T_PBYTE)p)+EC_ECINITCMDDESC_OFFS_RETRIES),
        wVal
              );
}

EC_INLINE(EC_T_VOID EC_ECINITCMDDESC_SET_NEWCYCLE(PEcInitCmdDesc p, EC_T_WORD wVal))
{
    if(wVal)
    {
        p->byFlags |= (1<<0);
    }
    else
    {
        
        p->byFlags &= ~(1<<0);
    }
}

EC_INLINE(EC_T_VOID EC_ECINITCMDDESC_SET_NEWFRAME(PEcInitCmdDesc p, EC_T_WORD wVal))
{
    if(wVal)
    {
        p->byFlags |= (1<<1);
    }
    else
    {
        
        p->byFlags &= ~(1<<1);
    }
}

EC_INLINE(EC_T_VOID EC_ECINITCMDDESC_SET_VALIDATE(PEcInitCmdDesc p, EC_T_WORD wVal))
{
    if(wVal)
    {
        p->byFlags |= (1<<2);
    }
    else
    {
        p->byFlags &= ~(1<<2);
    }
}

EC_INLINE(EC_T_VOID EC_ECINITCMDDESC_SET_VALIDATEMASK(PEcInitCmdDesc p, EC_T_WORD wVal))
{
    if(wVal)
    {
        p->byFlags |= (1<<3);
    }
    else
    {
        p->byFlags &= ~(1<<3);
    }
}

EC_INLINE(EC_T_VOID EC_ECINITCMDDESC_SET_CNT(PEcInitCmdDesc p, EC_T_WORD wVal))
{
    EC_SETWORD(
        (((EC_T_PBYTE)p)+EC_ECINITCMDDESC_OFFS_CNT),
        wVal
              );
}

#define SIZEOF_EcInitCmdDesc(p) (sizeof(EcInitCmdDesc) + EC_ICMDHDR_GET_LEN_LEN(&(((PEcInitCmdDesc)(p))->EcCmdHeader)) + \
    (EC_ECINITCMDDESC_GET_VALIDATE(((PEcInitCmdDesc)(p))) ? EC_ICMDHDR_GET_LEN_LEN(&(((PEcInitCmdDesc)(p))->EcCmdHeader)) : 0) + \
    (EC_ECINITCMDDESC_GET_VALIDATEMASK(((PEcInitCmdDesc)(p))) ? EC_ICMDHDR_GET_LEN_LEN(&(((PEcInitCmdDesc)(p))->EcCmdHeader)) : 0) + EC_ECINITCMDDESC_GET_CMTLEN(((PEcInitCmdDesc)(p))) + 1)
#define NEXT_EcInitCmdDesc(p)   (PEcInitCmdDesc)&(((EC_T_BYTE*)(p))[(sizeof(EcInitCmdDesc) + EC_ICMDHDR_GET_LEN_LEN(&(((PEcInitCmdDesc)(p))->EcCmdHeader)) + \
    (EC_ECINITCMDDESC_GET_VALIDATE(((PEcInitCmdDesc)(p))) ? EC_ICMDHDR_GET_LEN_LEN(&(((PEcInitCmdDesc)(p))->EcCmdHeader)) : 0) + \
    (EC_ECINITCMDDESC_GET_VALIDATEMASK(((PEcInitCmdDesc)(p))) ? EC_ICMDHDR_GET_LEN_LEN(&(((PEcInitCmdDesc)(p))->EcCmdHeader)) : 0) + EC_ECINITCMDDESC_GET_CMTLEN(((PEcInitCmdDesc)(p))) + 1)])
#define EcInitCmdDescData(p)    (&(((EC_T_BYTE*)(p))[sizeof(EcInitCmdDesc)]))
#define EcInitCmdDescVData(p)   (&(((EC_T_BYTE*)(p))[sizeof(EcInitCmdDesc) + EC_ICMDHDR_GET_LEN_LEN(&(((PEcInitCmdDesc)(p))->EcCmdHeader))]))
#define EcInitCmdDescVMData(p)  (&(((EC_T_BYTE*)(p))[sizeof(EcInitCmdDesc) + 2*EC_ICMDHDR_GET_LEN_LEN(&(((PEcInitCmdDesc)(p))->EcCmdHeader))]))
#define EcInitCmdDescComment(p) ((EC_T_CHAR*)&(((EC_T_BYTE*)(p))[sizeof(EcInitCmdDesc) + EC_ICMDHDR_GET_LEN_LEN(&(((PEcInitCmdDesc)(p))->EcCmdHeader)) + \
    (EC_ECINITCMDDESC_GET_VALIDATE(((PEcInitCmdDesc)(p))) ? EC_ICMDHDR_GET_LEN_LEN(&(((PEcInitCmdDesc)(p))->EcCmdHeader)) : 0) + \
    (EC_ECINITCMDDESC_GET_VALIDATEMASK(((PEcInitCmdDesc)(p))) ? EC_ICMDHDR_GET_LEN_LEN(&(((PEcInitCmdDesc)(p))->EcCmdHeader)) : 0)]))

/*---------------------------------------------------------------------------*/
#define ECCANOPENCMDDESC_TYPE_AUTO          0
#define ECCANOPENCMDDESC_TYPE_DD            1
#define ECCANOPENCMDDESC_TYPE_USER          2

/*---------------------------------------------------------------------------*/
/* Mailbox */
extern EC_T_CHAR* g_cStrMbxTypeText[];

#define ETHERCAT_MBOX_TYPE_ERROR        0       /* mailbox error response */
#define ETHERCAT_MBOX_TYPE_ADS          1       /* AMS/ADS header follows */
#define ETHERCAT_MBOX_TYPE_ETHERNET     2       /* ETHERCAT_ETHERNET_HEADER follows */
#define ETHERCAT_MBOX_TYPE_CANOPEN      3       /* ETHERCAT_CANOPEN_HEADER follows */
#define ETHERCAT_MBOX_TYPE_FILEACCESS   4       /* ETHERCAT_FOE_HEADER follows */
#ifdef INCLUDE_SOE_SUPPORT
#define ETHERCAT_MBOX_TYPE_SOE          5       /* ETHERCAT_SOE_HEADER follows */
#endif
#define ETHERCAT_MBOX_TYPE_VOE          15      /* ETHERCAT_VOE_HEADER follows */

#define EC_ECMBOXHDR_OFFS_LENGTH        ((EC_T_BYTE)0)
#define EC_ECMBOXHDR_OFFS_ADDRESS       ((EC_T_BYTE)2)
#define EC_ECMBOXHDR_OFFS_CHANPRIO      ((EC_T_BYTE)4)
#define EC_ECMBOXHDR_OFFS_TYPCRT        ((EC_T_BYTE)5)

#ifdef EC_NO_BITFIELDS

typedef struct TETHERCAT_MBOX_HEADER
{
    EC_T_WORD   __Length;               /* 0 */     /* following bytes */
    EC_T_WORD   __Address;              /* 2 */     /* S->M: phys addr of destination; M->S: phys addr of source; 0 = master */
    EC_T_WORD   __ChnPriTypCntRsvd;     /* 4 */     /* channel, Priority, wMbxType, Counter, Rsvd */
} EC_PACKED1 ETHERCAT_MBOX_HEADER, *PETHERCAT_MBOX_HEADER;
#define ETHERCAT_MBOX_HEADER_LEN    sizeof(ETHERCAT_MBOX_HEADER)
#define MBXHDR_CLR(pMbx) OsDbgAssert(sizeof(*pMbx)==sizeof(ETHERCAT_MBOX_HEADER)); OsMemset(pMbx,0,sizeof(ETHERCAT_MBOX_HEADER))

#define EC_ECMBOXHDR_SET_LENGTH(p, wVal)    ((p)->__Length = SWAPWORD(wVal))
#define EC_ECMBOXHDR_GET_LENGTH(p)          SWAPWORD((p)->__Length)
#define EC_ECMBOXHDR_SET_MBXTYPE(p, wVal)   EC_SET_FRM_WORD_BITFIELD((p)->__ChnPriTypCntRsvd,wVal,8,4)
#define EC_ECMBOXHDR_GET_MBXTYPE(p)         EC_GET_FRM_WORD_BITFIELD((p)->__ChnPriTypCntRsvd,8,4)
#define EC_ECMBOXHDR_SET_COUNTER(p, wVal)   EC_SET_FRM_WORD_BITFIELD((p)->__ChnPriTypCntRsvd,wVal,12,3)
#define EC_ECMBOXHDR_GET_COUNTER(p)         EC_GET_FRM_WORD_BITFIELD((p)->__ChnPriTypCntRsvd,12,3)

#else /* #ifdef EC_NO_BITFIELDS */


typedef struct TETHERCAT_MBOX_HEADER
{
    EC_T_WORD   __Length;               /* 0 */     /* following bytes */
    EC_T_WORD   __Address;              /* 2 */     /* S->M: phys addr of destination; M->S: phys addr of source; 0 = master */
    EC_T_WORD   __Channel         : 6;  /* 4.0 */   /* optional communication channels (default = 0) */
    EC_T_WORD   __Priority        : 2;  /* 4.6 */   /* optional communication priority (default = 0) */
    EC_T_WORD   __wMbxType        : 4;  /* 5.0 */   /* TETHERCAT_MBOX_TYPE_xxx */
    EC_T_WORD   __Counter         : 3;  /* 5.4 */   /* counter of the mailbox services (0 is start value, next value after 7 is 1) */
    EC_T_WORD   Rsvd              : 1;            /* unsupported protocol detected */
} EC_PACKED1 ETHERCAT_MBOX_HEADER, *PETHERCAT_MBOX_HEADER;
#define ETHERCAT_MBOX_HEADER_LEN    sizeof(ETHERCAT_MBOX_HEADER)
#define MBXHDR_CLR(pMbx) OsDbgAssert(sizeof(*pMbx)==sizeof(ETHERCAT_MBOX_HEADER)); OsMemset(pMbx,0,sizeof(ETHERCAT_MBOX_HEADER))


#define EC_ECMBOXHDR_SET_LENGTH(p, wVal)    ((p)->__Length = (wVal))
#define EC_ECMBOXHDR_GET_LENGTH(p)          ((p)->__Length)
#define EC_ECMBOXHDR_SET_MBXTYPE(p, wVal)   ((p)->__wMbxType  = (wVal))
#define EC_ECMBOXHDR_GET_MBXTYPE(p)         ((p)->__wMbxType)
#define EC_ECMBOXHDR_SET_COUNTER(p, wVal)   ((p)->__Counter  = (wVal))
#define EC_ECMBOXHDR_GET_COUNTER(p)         ((p)->__Counter)

#endif /* #else EC_NO_BITFIELDS */

typedef struct TETHERCAT_MBOX_CMD
{
    ETYPE_EC_CMD_HEADER     EcCmdHeader;
    ETHERCAT_MBOX_HEADER    MBoxHeader;
} EC_PACKED1 ETHERCAT_MBOX_CMD, *PETHERCAT_MBOX_CMD;
#define ETHERCAT_MBOX_CMD_LEN   sizeof(ETHERCAT_MBOX_CMD)
          

typedef struct TETHERCAT_EOE_TIMESTAMP
{
    EC_T_DWORD  TimeStamp;  /* 32 bit time stamp */
} ETHERCAT_EOE_TIMESTAMP, *PETHERCAT_EOE_TIMESTAMP;
#define	ETHERCAT_EOE_TIMESTAMP_LEN	sizeof(ETHERCAT_EOE_TIMESTAMP)

/*---------------------------------------------------------------------------*/
/* EoE (Ethernet over EtherCAT) */
#define ETHERCAT_ETHERNET_FRAME_TYPE_EOE_FRAG_REQ   0   /* EoE fragment request */
#define ETHERCAT_ETHERNET_FRAME_TYPE_EOE_REQ        1   /* EoE request */
#define ETHERCAT_ETHERNET_FRAME_TYPE_EOE_RSP        3   /* EoE response */
#define ETHERCAT_ETHERNET_FRAME_TYPE_EOE_SETIP_REQ  2   /* EoE set IP parameter request */
#define ETHERCAT_ETHERNET_FRAME_TYPE_EOE_SETIP_RSP  3   /* EoE set IP parameter response */
#define ETHERCAT_ETHERNET_FRAME_TYPE_EOE_SETAF_REQ  4   /* EoE set adress filter request */
#define ETHERCAT_ETHERNET_FRAME_TYPE_EOE_SETAF_RSP  5   /* EoE set adress filter response */

#define ETHERNET_FRAGMENT_GRANULAR      32              /* length of each fragment (except the last fragment) must be dividable by 32 */
#define ETHERNET_FRAGMENT_MASK          0xFFFFFFE0
#define ETHERNET_FRAGMENT_BUFFER(n)     ((n+31)/32)
#define ETHERNET_FRAMENUMBER_MASK       0x0000000F
#define ETHERNET_MAX_FRAGMENTS          48
#define ETHERNET_MAX_FRAGMENTBUFFER (ETHERNET_MAX_FRAGMENTS*ETHERNET_FRAGMENT_GRANULAR) /* 1536 */

#define EC_ECETHHDR_OFFS_FRAMETYPEPORT      ((EC_T_BYTE)0)
#define EC_ECETHHDR_OFFS_FRAGTIME           ((EC_T_BYTE)1)
#define EC_ECETHHDR_OFFS_RESULT             ((EC_T_BYTE)2)

typedef struct TETHERCAT_ETHERNET_HEADER
{
#ifdef EC_BIG_ENDIAN
    EC_T_WORD   Reserved            : 5;                        
    EC_T_WORD   __TimeRequest       : 1;            /*  1.2 */  /* EC_TRUE if time stamp value of the send time is requested */
    EC_T_WORD   __TimeAppended      : 1;            /*  1.1 */  /* EC_TRUE if time stamp will be appended after the EoE data in the last fragment */
    EC_T_WORD   __LastFragment      : 1;            /*  1.0 */  /* EC_TRUE if last fragment */
    EC_T_WORD   __Port              : 4;            /*  0.4 */  /* port number */
    EC_T_WORD   __FrameType         : 4;            /*  0.0 */  /* frame type */
#else
    EC_T_WORD   __FrameType         : 4;            /*  0.0 */  /* frame type */
    EC_T_WORD   __Port              : 4;            /*  0.4 */  /* port number */
    EC_T_WORD   __LastFragment      : 1;            /*  1.0 */  /* EC_TRUE if last fragment */
    EC_T_WORD   __TimeAppended      : 1;            /*  1.1 */  /* EC_TRUE if time stamp will be appended after the EoE data in the last fragment */
    EC_T_WORD   __TimeRequest       : 1;            /*  1.2 */  /* EC_TRUE if time stamp value of the send time is requested */
    EC_T_WORD   Reserved            : 5;                        
#endif
    union _t_uFragment
    {                                                           
        struct _t_sFragCompSz
        {                                                       
#ifdef EC_BIG_ENDIAN
            EC_T_WORD   __FrameNumber       : 4;    /*  2.12*/  /* number of the Ethernet frame */
            EC_T_WORD   __CompleteFrameSize : 6;    /*  2.6 */  /* (complete size of the Ethernet frame + 17)/32 */
            EC_T_WORD   __FragmentNumber    : 6;    /*  2.0 */  /* fragment number */
#else
            EC_T_WORD   __FragmentNumber    : 6;    /*  2.0 */  /* fragment number */
            EC_T_WORD   __CompleteFrameSize : 6;    /*  2.6 */  /* (complete size of the Ethernet frame + 17)/32 */
            EC_T_WORD   __FrameNumber       : 4;    /*  2.12*/  /* number of the Ethernet frame */
#endif
        } EC_PACKED1 FragCompSz;                               
        struct _t_sFragOffBuf
        {                                                     
#ifdef EC_BIG_ENDIAN
            EC_T_WORD   __FrameNumber       : 4;    /*  2.12*/  /* number of the Ethernet frame */
            EC_T_WORD   __OffsetBuffer      : 6;    /*  2.6 */  /* byte offset multiplied by 32 (if Fragment != 0);  */
                                                              /* buffer size multiplied by 32 (if Fragment == 0) */
            EC_T_WORD   __FragmentNumber    : 6;    /*  2.0 */  /* fragment number */
#else
            EC_T_WORD   __FragmentNumber    : 6;    /*  2.0 */  /* fragment number */
            EC_T_WORD   __OffsetBuffer      : 6;    /*  2.6 */  /* byte offset multiplied by 32 (if Fragment != 0);  */
                                                              /* buffer size multiplied by 32 (if Fragment == 0) */
            EC_T_WORD   __FrameNumber       : 4;    /*  2.12*/  /* number of the Ethernet frame */
#endif
        } EC_PACKED1 FragOffBuf;                               
        EC_T_WORD __Result                  :16;    /*  2 */    /* result (e.g. for set IP request) */
    } EC_PACKED1 uFragment;
} EC_PACKED1 ETHERCAT_ETHERNET_HEADER, *PETHERCAT_ETHERNET_HEADER;
#define ETHERCAT_ETHERNET_HEADER_LEN    sizeof(ETHERCAT_ETHERNET_HEADER)
#define EOEHDR_CLR(pEoe) OsDbgAssert(sizeof(*pEoe)==sizeof(ETHERCAT_ETHERNET_HEADER)); OsMemset(pEoe,0,sizeof(ETHERCAT_ETHERNET_HEADER))

EC_INLINE(EC_T_WORD EC_ECETHHDR_GET_FRAMETYPE(PETHERCAT_ETHERNET_HEADER p))
{
    return (EC_T_WORD)
           (
        (*(((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_FRAMETYPEPORT))
        &0xf
           );
}
EC_INLINE(EC_T_WORD EC_ECETHHDR_GET_PORT(PETHERCAT_ETHERNET_HEADER p))
{
    return (EC_T_WORD)
           (
        (
        (*(((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_FRAMETYPEPORT))
        >>4
        )
        &0xf
           );
}
EC_INLINE(EC_T_WORD EC_ECETHHDR_GET_LASTFRAGMENT(PETHERCAT_ETHERNET_HEADER p))
{
    return (EC_T_WORD)
           (
        (
        (*(((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_FRAGTIME))
        >>0
        )
        &0x1
        );
}
EC_INLINE(EC_T_WORD EC_ECETHHDR_GET_TIMEAPPENDED(PETHERCAT_ETHERNET_HEADER p))
{
    return (EC_T_WORD)
           (
        (
        (*(((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_FRAGTIME))
        >>1
        )
        &0x1
        );
}
EC_INLINE(EC_T_WORD EC_ECETHHDR_GET_TIMEREQUEST(PETHERCAT_ETHERNET_HEADER p))
{
    return (EC_T_WORD)
           (
        (
        (*(((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_FRAGTIME))
        >>2
        )
        &0x1
        );
}
EC_INLINE(EC_T_WORD EC_ECETHHDR_GET_FRAGMENTNUMBER(PETHERCAT_ETHERNET_HEADER p))
{
    return (EC_T_WORD)
           (EC_GET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_RESULT)
                      )
            &((1<<6)-1)
           );
}
EC_INLINE(EC_T_WORD EC_ECETHHDR_GET_COMPLETEFRAMESIZE(PETHERCAT_ETHERNET_HEADER p))
{
    return (EC_T_WORD)
           ((EC_GET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_RESULT)
                      )
            >>6
            )
        &((1<<6)-1)
           );
}
EC_INLINE(EC_T_WORD EC_ECETHHDR_GET_OFFSETBUFFER(PETHERCAT_ETHERNET_HEADER p))
{
    return EC_ECETHHDR_GET_COMPLETEFRAMESIZE(p);
}
EC_INLINE(EC_T_WORD EC_ECETHHDR_GET_FRAMENUMBER(PETHERCAT_ETHERNET_HEADER p))
{
    return (EC_T_WORD)
           ((EC_GET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_RESULT)
                       )
            >>12
            )
           &0xf
           );
}
EC_INLINE(EC_T_WORD EC_ECETHHDR_GET_RESULT(PETHERCAT_ETHERNET_HEADER p))
{
    return EC_GET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_RESULT)
                     );
}

EC_INLINE(EC_T_VOID EC_ECETHHDR_SET_FRAGMENTNUMBER(PETHERCAT_ETHERNET_HEADER p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_RESULT), 
        ((EC_T_WORD)((EC_GET_FRM_WORD((((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_RESULT))&(~((1<<6)-1)) )|(wVal&((1<<6)-1))))
              );
}
EC_INLINE(EC_T_VOID EC_ECETHHDR_SET_COMPLETEFRAMESIZE(PETHERCAT_ETHERNET_HEADER p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_RESULT),
        ((EC_T_WORD)((EC_GET_FRM_WORD((((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_RESULT)) & (~(((1<<6)-1)<<6))) | ((wVal & ((1<<6)-1))<<6)))
              );
}
EC_INLINE(EC_T_VOID EC_ECETHHDR_SET_OFFSETBUFFER(PETHERCAT_ETHERNET_HEADER p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_RESULT),
        ((EC_T_WORD)((EC_GET_FRM_WORD((((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_RESULT)) & (~(((1<<6)-1)<<6))) | ((wVal & ((1<<6)-1)))<<6) )
              );
}
EC_INLINE(EC_T_VOID EC_ECETHHDR_SET_FRAMENUMBER(PETHERCAT_ETHERNET_HEADER p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_RESULT),
        ((EC_T_WORD)((EC_GET_FRM_WORD((((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_RESULT)) & (~(0xf<<12))) | ((wVal&0xf)<<12)))
              );
}
EC_INLINE(EC_T_VOID EC_ECETHHDR_SET_LASTFRAGMENT(PETHERCAT_ETHERNET_HEADER p, EC_T_WORD wVal))
{
    (*(((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_FRAGTIME)) = (EC_T_BYTE)
        (
        (((*(((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_FRAGTIME))&(~(1<<0))) | ((wVal&1)<<0))
        );
}
EC_INLINE(EC_T_VOID EC_ECETHHDR_SET_TIMEAPPENDED(PETHERCAT_ETHERNET_HEADER p, EC_T_WORD wVal))
{
    (*(((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_FRAGTIME)) = (EC_T_BYTE)
        (
        (((*(((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_FRAGTIME))&(~(1<<1))) | ((wVal&1)<<1))
        );
}
EC_INLINE(EC_T_VOID EC_ECETHHDR_SET_TIMEREQUEST(PETHERCAT_ETHERNET_HEADER p, EC_T_WORD wVal))
{
    (*(((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_FRAGTIME)) = (EC_T_BYTE)
        (
        (((*(((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_FRAGTIME))&(~(1<<2))) | ((wVal&1)<<2))
        );
}
EC_INLINE(EC_T_VOID EC_ECETHHDR_SET_FRAMETYPE(PETHERCAT_ETHERNET_HEADER p, EC_T_WORD wVal))
{
    (*(((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_FRAMETYPEPORT)) = (EC_T_BYTE)
        (
        (((*(((EC_T_PBYTE)p)+EC_ECETHHDR_OFFS_FRAMETYPEPORT))&(~(0xf<<0))) | ((wVal&0xf)<<0))
        );    
}



/*---------------------------------------------------------------------------*/
/* CoE (CANopen over EtherCAT) */
#define ETHERCAT_CANOPEN_TYPE_EMERGENCY 1
#define ETHERCAT_CANOPEN_TYPE_SDOREQ    2
#define ETHERCAT_CANOPEN_TYPE_SDORES    3
#define ETHERCAT_CANOPEN_TYPE_TXPDO     4
#define ETHERCAT_CANOPEN_TYPE_RXPDO     5
#define ETHERCAT_CANOPEN_TYPE_TXPDO_RTR 6       /* Remote transmission request of TXPDO (master requested) */
#define ETHERCAT_CANOPEN_TYPE_RXPDO_RTR 7       /* Remote transmission request of RXPDO (slave requested) */
#define ETHERCAT_CANOPEN_TYPE_SDOINFO   8

#ifdef EC_NO_BITFIELDS

typedef EC_T_WORD ETHERCAT_CANOPEN_HEADER, *PETHERCAT_CANOPEN_HEADER;
#define ETHERCAT_CANOPEN_HEADER_LEN sizeof(ETHERCAT_CANOPEN_HEADER)
#define COEHDR_CLR(pCan) OsDbgAssert(sizeof(*pCan)==ETHERCAT_CANOPEN_HEADER_LEN); (*(pCan) = 0)

#define EC_ECCOEHDR_GET_COENUMBER(p)            EC_GET_FRM_WORD_BITFIELD(*(p),0,9)
#define EC_ECCOEHDR_SET_COENUMBER(p, wVal)      EC_SET_FRM_WORD_BITFIELD(*(p),wVal,0,9)

#define EC_ECCOEHDR_GET_COETYPE(p)              EC_GET_FRM_WORD_BITFIELD(*(p),12,4)
#define EC_ECCOEHDR_SET_COETYPE(p, wVal)        EC_SET_FRM_WORD_BITFIELD(*(p),wVal,12,4)

#else /* #ifdef EC_NO_BITFIELDS */

typedef struct TETHERCAT_CANOPEN_HEADER
{
    union _t_uCoeHdr
    {
        EC_T_WORD __wCoeHdr:16;
        struct _t_swCoeHdr
        {
            EC_T_WORD   __wCoeNumber        : 9;        /* e.g. PDO number */
            EC_T_WORD   Reserved            : 3;        /* = 0 */
            EC_T_WORD   __wCoeType          : 4;        /* CANopen type */
        } EC_PACKED1 swCoeHdr;
    } EC_PACKED1 uCoeHdr;
           
} EC_PACKED1 ETHERCAT_CANOPEN_HEADER, *PETHERCAT_CANOPEN_HEADER;
#define ETHERCAT_CANOPEN_HEADER_LEN sizeof(ETHERCAT_CANOPEN_HEADER)
#define COEHDR_CLR(pCan) OsDbgAssert(sizeof(*pCan)==ETHERCAT_CANOPEN_HEADER_LEN); ((pCan)->uCoeHdr.__wCoeHdr = 0)

#define EC_ECCOEHDR_SET_COENUMBER(p, wVal)      ((p)->uCoeHdr.swCoeHdr.__wCoeNumber = (wVal))  
#define EC_ECCOEHDR_GET_COETYPE(p)              ((p)->uCoeHdr.swCoeHdr.__wCoeType)
#define EC_ECCOEHDR_SET_COETYPE(p, wVal)        ((p)->uCoeHdr.swCoeHdr.__wCoeType = (wVal))

#endif /* #else EC_NO_BITFIELDS */


#define EC_ECSDOHDR_OFFS_INDEX      ((EC_T_BYTE)1)
#define EC_ECSDOHDR_OFFS_SUBINDEX   ((EC_T_BYTE)3)
#define EC_ECSDOHDR_OFFS_SDODATA    ((EC_T_BYTE)4)

typedef struct TETHERCAT_SDO_HEADER
{
    union _t_uHdr_sdoheader
    {
        struct _t_sIdq
        {   /* Initiate Download Request */
#ifdef EC_BIG_ENDIAN
            EC_T_BYTE   Ccs         : 3;    /* = 1 */
            EC_T_BYTE   Complete    : 1;
            EC_T_BYTE   Size        : 2;
            EC_T_BYTE   Expedited   : 1;
            EC_T_BYTE   SizeInd     : 1;
#else
            EC_T_BYTE   SizeInd     : 1;
            EC_T_BYTE   Expedited   : 1;
            EC_T_BYTE   Size        : 2;
            EC_T_BYTE   Complete    : 1;
            EC_T_BYTE   Ccs         : 3;    /* = 1 */
#endif
        } EC_PACKED1 Idq;
        struct _t_sIds
        {   /* Initiate Download Response */
#ifdef EC_BIG_ENDIAN
            EC_T_BYTE   Scs         : 3;    /* = 3 */
            EC_T_BYTE   Reserved    : 5;
#else
            EC_T_BYTE   Reserved    : 5;
            EC_T_BYTE   Scs         : 3;    /* = 3 */
#endif
        } EC_PACKED1 Ids;
        struct _t_sDsq
        {   /* Download Segment Request */
#ifdef EC_BIG_ENDIAN
            EC_T_BYTE   Ccs         : 3;    /* = 0 */
            EC_T_BYTE   Toggle      : 1;
            EC_T_BYTE   Size        : 3;
            EC_T_BYTE   LastSeg     : 1;
#else
            EC_T_BYTE   LastSeg     : 1;
            EC_T_BYTE   Size        : 3;
            EC_T_BYTE   Toggle      : 1;
            EC_T_BYTE   Ccs         : 3;    /* = 0 */
#endif
        } EC_PACKED1 Dsq;
        struct _t_sDss
        {   /* Download Segment Response */
#ifdef EC_BIG_ENDIAN
            EC_T_BYTE   Scs         : 3;    /* = 1 */
            EC_T_BYTE   Toggle      : 1;
            EC_T_BYTE   Reserved    : 4;
#else
            EC_T_BYTE   Reserved    : 4;
            EC_T_BYTE   Toggle      : 1;
            EC_T_BYTE   Scs         : 3;    /* = 1 */
#endif
        } EC_PACKED1 Dss;
        struct _t_sIuq
        {   /* Initiate Upload Request */
#ifdef EC_BIG_ENDIAN
            EC_T_BYTE   Ccs         : 3;    /* = 2 */
            EC_T_BYTE   Complete    : 1;
            EC_T_BYTE   Reserved    : 4;
#else
            EC_T_BYTE   Reserved    : 4;
            EC_T_BYTE   Complete    : 1;
            EC_T_BYTE   Ccs         : 3;    /* = 2 */
#endif
        } EC_PACKED1 Iuq;
        struct _t_sIus
        {   /* Initiate Upload Response */
#ifdef EC_BIG_ENDIAN
            EC_T_BYTE   Scs         : 3;    /* = 2 */
            EC_T_BYTE   Reserved    : 1;
            EC_T_BYTE   Size        : 2;
            EC_T_BYTE   Expedited   : 1;
            EC_T_BYTE   SizeInd     : 1;
#else
            EC_T_BYTE   SizeInd     : 1;
            EC_T_BYTE   Expedited   : 1;
            EC_T_BYTE   Size        : 2;
            EC_T_BYTE   Reserved    : 1;
            EC_T_BYTE   Scs         : 3;    /* = 2 */
#endif
        } EC_PACKED1 Ius;
        struct _t_sUsq
        {   /* Upload Segment Request */
#ifdef EC_BIG_ENDIAN
            EC_T_BYTE   Ccs         : 3;    /* = 3 */
            EC_T_BYTE   Toggle      : 1;
            EC_T_BYTE   Reserved    : 4;
#else
            EC_T_BYTE   Reserved    : 4;
            EC_T_BYTE   Toggle      : 1;
            EC_T_BYTE   Ccs         : 3;    /* = 3 */
#endif
        } EC_PACKED1 Usq;
        struct _t_sUss
        {   /* Upload Segment Response */
#ifdef EC_BIG_ENDIAN
            EC_T_BYTE   Scs         : 3;    /* = 0 */
            EC_T_BYTE   Toggle      : 1;
            EC_T_BYTE   Size        : 3;
            EC_T_BYTE   LastSeg     : 1;
#else
            EC_T_BYTE   LastSeg     : 1;
            EC_T_BYTE   Size        : 3;
            EC_T_BYTE   Toggle      : 1;
            EC_T_BYTE   Scs         : 3;    /* = 0 */
#endif
        } EC_PACKED1 Uss;
        struct _t_sCS
        {   /* Abort Transfer */
#ifdef EC_BIG_ENDIAN
            EC_T_BYTE   Ccs         : 3;    /* = 4 */
            EC_T_BYTE   Reserved    : 5;
#else
            EC_T_BYTE   Reserved    : 5;
            EC_T_BYTE   Ccs         : 3;    /* = 4 */
#endif
        } EC_PACKED1 Abt;
        EC_T_BYTE CS;                                
    } EC_PACKED1 uHdr;                                   /* 0 */
    EC_T_WORD   __Index;                                /* 1 */
    EC_T_BYTE   SubIndex;                               /* 3 */
    EC_T_DWORD  __dwSdoData;                            /* 4 */
} EC_PACKED1 ETHERCAT_SDO_HEADER, *PETHERCAT_SDO_HEADER;

#define SDO_HDR_INDEX_OFFSET            (sizeof(EC_T_BYTE))                                     /* CS */
#define SDO_HDR_SUB_INDEX_OFFSET        (sizeof(EC_T_BYTE)+sizeof(EC_T_WORD))                   /* CS + Index */
#define SDO_HDR_DATA_OFFSET             (sizeof(EC_T_BYTE)+sizeof(EC_T_WORD)+sizeof(EC_T_BYTE)) /* CS + Index + SubIndex */
#define ETHERCAT_SDO_HEADER_LEN         sizeof(ETHERCAT_SDO_HEADER)
#define ETHERCAT_MIN_SDO_MBOX_LEN       (ETHERCAT_MBOX_HEADER_LEN + ETHERCAT_CANOPEN_HEADER_LEN + ETHERCAT_SDO_HEADER_LEN)
#define SDOHDR_CLR(pSdo) OsDbgAssert(sizeof(*pSdo)==ETHERCAT_SDO_HEADER_LEN); OsMemset(pSdo,0,ETHERCAT_SDO_HEADER_LEN)

EC_INLINE(EC_T_WORD EC_ECSDOHDR_GET_INDEX(PETHERCAT_SDO_HEADER p))
{
    return EC_GET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECSDOHDR_OFFS_INDEX)
                     );
}
EC_INLINE(EC_T_DWORD EC_ECSDOHDR_GET_SDODATA(PETHERCAT_SDO_HEADER p))
{
    return EC_GET_FRM_DWORD(
        (((EC_T_PBYTE)p)+EC_ECSDOHDR_OFFS_SDODATA)
                      );
}
EC_INLINE(EC_T_VOID EC_ECSDOHDR_SET_INDEX(PETHERCAT_SDO_HEADER p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECSDOHDR_OFFS_INDEX),
        wVal
              );
}
EC_INLINE(EC_T_VOID EC_ECSDOHDR_SET_SDODATA(PETHERCAT_SDO_HEADER p, EC_T_DWORD dwVal))
{
    EC_SET_FRM_DWORD(
        (((EC_T_PBYTE)p)+EC_ECSDOHDR_OFFS_SDODATA),
        dwVal
               );
}

EC_INLINE(EC_T_PBYTE EC_ECSDOHDR_SDODATA(PETHERCAT_SDO_HEADER p))
{
    return (EC_T_PBYTE)(((EC_T_PBYTE)p)+EC_ECSDOHDR_OFFS_SDODATA);
}

/* command specifier - request */
#define SDO_CCS_DOWNLOAD_SEGMENT                        0
#define SDO_CCS_INITIATE_DOWNLOAD                       1
#define SDO_CCS_INITIATE_UPLOAD                         2
#define SDO_CCS_UPLOAD_SEGMENT                          3
#define SDO_CCS_ABORT_TRANSFER                          4

/* command specifier - response */                                                       
#define SDO_SCS_UPLOAD_SEGMENT                          0
#define SDO_SCS_DOWNLOAD_SEGMENT                        1
#define SDO_SCS_INITIATE_UPLOAD                         2
#define SDO_SCS_INITIATE_DOWNLOAD                       3
                                                        
#define SDO_DOWNLOAD_SEGMENT_MAX_DATA                   7
                                                        
#define SDO_ABORTCODE_TOGGLE                            0x05030000  /* Toggle bit not alternated. */
#define SDO_ABORTCODE_TIMEOUT                           0x05040000  /* SDO protocol timed out. */
#define SDO_ABORTCODE_CCS_SCS                           0x05040001  /* Client/server command specifier not valid or unknown. */
#define SDO_ABORTCODE_BLK_SIZE                          0x05040002  /* Invalid block size (block mode only). */
#define SDO_ABORTCODE_SEQNO                             0x05040003  /* Invalid sequence number (block mode only). */
#define SDO_ABORTCODE_CRC                               0x05040004  /* CRC error (block mode only). */
#define SDO_ABORTCODE_MEMORY                            0x05040005  /* Out of memory.*/
#define SDO_ABORTCODE_ACCESS                            0x06010000  /* Unsupported access to an object.*/
#define SDO_ABORTCODE_WRITEONLY                         0x06010001  /* Attempt to read a write only object.*/
#define SDO_ABORTCODE_READONLY                          0x06010002  /* Attempt to write a read only object.*/
#define SDO_ABORTCODE_INDEX                             0x06020000  /* Object does not exist in the object dictionary.*/
#define SDO_ABORTCODE_PDO_MAP                           0x06040041  /* Object cannot be mapped to the PDO.*/
#define SDO_ABORTCODE_PDO_LEN                           0x06040042  /* The number and length of the objects to be mapped would exceed PDO length.*/
#define SDO_ABORTCODE_P_INCOMP                          0x06040043  /* General parameter incompatibility reason.*/
#define SDO_ABORTCODE_I_INCOMP                          0x06040047  /* General internal incompatibility in the device.*/
#define SDO_ABORTCODE_HARDWARE                          0x06060000  /* Access failed due to an hardware error.*/
#define SDO_ABORTCODE_DATA_SIZE                         0x06070010  /* Data type does not match, length of service parameter does not match*/
#define SDO_ABORTCODE_DATA_SIZE1                        0x06070012  /* Data type does not match, length of service parameter too high*/
#define SDO_ABORTCODE_DATA_SIZE2                        0x06070013  /* Data type does not match, length of service parameter too low*/
#define SDO_ABORTCODE_OFFSET                            0x06090011  /* Sub-index does not exist.*/
#define SDO_ABORTCODE_DATA_RANGE                        0x06090030  /* Value range of parameter exceeded (only for write access).*/
#define SDO_ABORTCODE_DATA_RANGE1                       0x06090031  /* Value of parameter written too high.*/
#define SDO_ABORTCODE_DATA_RANGE2                       0x06090032  /* Value of parameter written too low.*/
#define SDO_ABORTCODE_MINMAX                            0x06090036  /* Maximum value is less than minimum value.*/
#define SDO_ABORTCODE_GENERAL                           0x08000000  /* general error*/
#define SDO_ABORTCODE_TRANSFER                          0x08000020  /* Data cannot be transferred or stored to the application.*/
#define SDO_ABORTCODE_TRANSFER1                         0x08000021  /* Data cannot be transferred or stored to the application because of local control.*/
#define SDO_ABORTCODE_TRANSFER2                         0x08000022  /* Data cannot be transferred or stored to the application because of the present device state.*/
#define SDO_ABORTCODE_DICTIONARY                        0x08000023  /* Object dictionary dynamic generation fails or no object dictionary is present (e.g. object dictionary is generated from file and generation fails because of an file error).*/

#define SDO_SUB_IDX_SUBINDEX_CNT                        0

#define SDO_IDX_DEVICE_TYPE                             0x1000
#define SDO_IDX_ERROR                                   0x1001
#define SDO_IDX_MANUFACTURER_NAME                       0x1008
#define SDO_IDX_MANUFACTURER_HW_VER                     0x1009
#define SDO_IDX_MANUFACTURER_SW_VER                     0x100A
#define SDO_IDX_IDENTITY_OBJECT                         0x1018
#define SDO_SIDX_IDENTITY_OBJECT_VENDORID               1
#define SDO_SIDX_IDENTITY_OBJECT_PCODE                  2
#define SDO_SIDX_IDENTITY_OBJECT_REVNO                  3
#define SDO_SIDX_IDENTITY_OBJECT_SERNO                  4

                                                        
#define SDO_IDX_ETHERCAT_ADDR                           0x1100

#define SDO_IDX_RXPDO1_PARA                             0x1400
#define SDO_SIDX_RXPDXX_PARA_EXCLUDE                    1

#define SDO_IDX_RXPDO2_PARA                             0x1401
/* ... */
#define SDO_IDX_RXPDO512_PARA                           0x15FF

#define SDO_IDX_TXPDO1_PARA                             0x1800
#define SDO_IDX_TXPDO2_PARA                             0x1801
/* ... */
#define SDO_IDX_TXPDO512_PARA                           0x19FF

#define SDO_IDX_RXPDO1_MAPPING                          0x1600
#define SDO_IDX_RXPDO2_MAPPING                          0x1601
/* ... */
#define SDO_IDX_RXPDO512_MAPPING                        0x17FF
                                                        
#define SDO_IDX_TXPDO1_MAPPING                          0x1A00
#define SDO_IDX_TXPDO2_MAPPING                          0x1A01
/* ... */
#define SDO_IDX_TXPDO512_MAPPING                        0x1BFF
                                                        
#define SDO_IDX_SYNCMAN_TYPE                            0x1C00
#define SDO_IDX_SYNCMAN0_PDOASSIGN                      0x1C10
#define SDO_IDX_SYNCMAN1_PDOASSIGN                      0x1C11
#define SDO_IDX_SYNCMAN2_PDOASSIGN                      0x1C12
#define SDO_IDX_SYNCMAN3_PDOASSIGN                      0x1C13

/* ... */
#define SDO_IDX_SYNCMAN31_PDOASSIGN                     0x1C2F

#define SDO_IDX_ECAT_MEMORY_0000_00FF                   0x1D00
#define SDO_IDX_ECAT_MEMORY_0100_01FF                   0x1D01
/* ... */
#define SDO_IDX_ECAT_MEMORY_FF00_FFFF                   0x1DFF

/* CoE SDO Information */
#define ECAT_COE_INFO_OPCODE_LIST_Q                     1
#define ECAT_COE_INFO_OPCODE_LIST_S                     2
#define ECAT_COE_INFO_OPCODE_OBJ_Q                      3
#define ECAT_COE_INFO_OPCODE_OBJ_S                      4
#define ECAT_COE_INFO_OPCODE_ENTRY_Q                    5
#define ECAT_COE_INFO_OPCODE_ENTRY_S                    6
#define ECAT_COE_INFO_OPCODE_ERROR_S                    7

#define ECAT_COE_INFO_LIST_TYPE_LENGTH                  0
#define ECAT_COE_INFO_LIST_TYPE_ALL                     1
#define ECAT_COE_INFO_LIST_TYPE_RXPDOMAP                2
#define ECAT_COE_INFO_LIST_TYPE_TXPDOMAP                3
#define ECAT_COE_INFO_LIST_TYPE_BACKUP                  4
                                                        
#define ECAT_COE_INFO_OBJCODE_NULL                      0
#define ECAT_COE_INFO_OBJCODE_DOMAIN                    2
#define ECAT_COE_INFO_OBJCODE_DEFTYPE                   5
#define ECAT_COE_INFO_OBJCODE_DEFSTRUCT                 6
#define ECAT_COE_INFO_OBJCODE_VAR                       7
#define ECAT_COE_INFO_OBJCODE_ARRAY                     8
#define ECAT_COE_INFO_OBJCODE_RECORD                    9
                                                        
#define ECAT_COE_INFO_OBJCAT_OPTIONAL                   0
#define ECAT_COE_INFO_OBJCAT_MANDATORY                  1
                                                        
#define ECAT_COE_INFO_OBJACCESS_RO                      0x07
#define ECAT_COE_INFO_OBJACCESS_RW                      0x3f

typedef struct TETHERCAT_SDO_INFO_LIST
{
    EC_T_WORD   __ListType;                   /* == SDO_INFO_LIST_TYPE_XXX */
    struct _t_sRes_sdoinfolist
    {
        EC_T_WORD   __Index[1];       
    } EC_PACKED1 Res;
} EC_PACKED1 ETHERCAT_SDO_INFO_LIST, *PETHERCAT_SDO_INFO_LIST;

EC_INLINE(EC_T_WORD EC_SDOINFOLIST_GET_LISTTYPE(PETHERCAT_SDO_INFO_LIST p))
{
    return EC_GET_FRM_WORD(
        (((EC_T_PBYTE)p))
                     );
}
EC_INLINE(EC_T_VOID EC_SDOINFOLIST_SET_LISTTYPE(PETHERCAT_SDO_INFO_LIST p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD(
        (((EC_T_PBYTE)p)),
        wVal
              );
}

#define EC_SDOINFOOBJ_OFFS_INDEX        ((EC_T_BYTE)0)
#define EC_SDOINFOOBJ_OFFS_DATATYPE     ((EC_T_BYTE)2)

typedef struct TETHERCAT_SDO_INFO_OBJ
{
    EC_T_WORD   __Index;
    struct _t_sRes_sdoinfoobj
    {
        EC_T_WORD   __DataType;               /* refer to data type index */
        EC_T_BYTE   MaxSubIndex;            /* max subIndex */
#ifdef EC_BIG_ENDIAN
        EC_T_BYTE   Reserved        : 3;    /* == 0 */
        EC_T_BYTE   ObjCategory     : 1;    /* 0=optional, 1=mandatory */
        EC_T_BYTE   ObjCode         : 4;    /* defined in DS 301 (Table 37) */
#else
        EC_T_BYTE   ObjCode         : 4;    /* defined in DS 301 (Table 37) */
        EC_T_BYTE   ObjCategory     : 1;    /* 0=optional, 1=mandatory */
        EC_T_BYTE   Reserved        : 3;    /* == 0 */
#endif
        EC_T_CHAR   Name[1];                /* rest of mailbox data */
    } EC_PACKED1 Res;
} EC_PACKED1 ETHERCAT_SDO_INFO_OBJ, *PETHERCAT_SDO_INFO_OBJ;

EC_INLINE(EC_T_WORD EC_SDOINFOOBJ_GET_INDEX(PETHERCAT_SDO_INFO_OBJ p))
{
    return EC_GET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_SDOINFOOBJ_OFFS_INDEX)
        );
}
EC_INLINE(EC_T_WORD EC_SDOINFOOBJ_GET_DATATYPE(PETHERCAT_SDO_INFO_OBJ p))
{
    return EC_GET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_SDOINFOOBJ_OFFS_DATATYPE)
        );
}
EC_INLINE(EC_T_VOID EC_SDOINFOOBJ_SET_INDEX(PETHERCAT_SDO_INFO_OBJ p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_SDOINFOOBJ_OFFS_INDEX),
        wVal
              );
}
EC_INLINE(EC_T_VOID EC_SDOINFOOBJ_SET_DATATYPE(PETHERCAT_SDO_INFO_OBJ p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_SDOINFOOBJ_OFFS_DATATYPE),
        wVal
              );
}

#define EC_ECSDOINFOENTRY_OFFS_INDEX        ((EC_T_BYTE)0)
#define EC_ECSDOINFOENTRY_OFFS_DATATYPE     ((EC_T_BYTE)4)
#define EC_ECSDOINFOENTRY_OFFS_BITLEN       ((EC_T_BYTE)6)
#define EC_ECSDOINFOENTRY_OFFS_BITPARM      ((EC_T_BYTE)8)

typedef struct TETHERCAT_SDO_INFO_ENTRY
{
    EC_T_WORD       __Index;
    EC_T_BYTE       SubIdx; 
    EC_T_BYTE       ValueInfo;              /* bit0 = ObjAccess, bit1 = ObjCategory, bit2 = PdoMapping, bit3 = UnitType */
                                            /* bit4 = DefaultValue, bit5 = MinValue, bit6 = MaxValue */
    struct _t_sRes_sdoinfoentry
    {
        EC_T_WORD   __DataType;             /* refer to data type index */
        EC_T_WORD   __BitLen;       
#ifdef EC_BIG_ENDIAN
        EC_T_BYTE   __TxPdoMapping  : 1;    /* */
        EC_T_BYTE   __RxPdoMapping  : 1;    /* */
        EC_T_BYTE   __ObjAccess     : 6;    /* bit0 = read; bit1 = write; bit2 = const. bit3 = PRE-OP; bit4 = SAFE-OP; bit5 = OP. */
        EC_T_BYTE   Reserved        : 8;    /* for future use */
#else
        EC_T_WORD   __ObjAccess     : 6;    /* bit0 = read; bit1 = write; bit2 = const. bit3 = PRE-OP; bit4 = SAFE-OP; bit5 = OP. */
        EC_T_WORD   __RxPdoMapping  : 1;    /* */
        EC_T_WORD   __TxPdoMapping  : 1;    /* */
        EC_T_WORD   Reserved        : 8;    /* for future use */
#endif
/*      EC_T_WORD   UnitType;               // optional if bit3 of valueInfo */
/*      EC_T_BYTE   DefaultValue[];         // optional if bit4 of valueInfo */
/*      EC_T_BYTE   MinValue[];             // optional if bit5 of valueInfo */
/*      EC_T_BYTE   MaxValue[];             // optional if bit6 of valueInfo */
/*      EC_T_CHAR   Desc[];                 // rest of mailbox data */
    } EC_PACKED1 Res;
} EC_PACKED1 ETHERCAT_SDO_INFO_ENTRY, *PETHERCAT_SDO_INFO_ENTRY;

EC_INLINE(EC_T_WORD EC_SDOINFOENTRY_GET_INDEX(PETHERCAT_SDO_INFO_ENTRY p))
{
    return EC_GET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECSDOINFOENTRY_OFFS_INDEX)
                     );
}
EC_INLINE(EC_T_WORD EC_SDOINFOENTRY_GET_DATATYPE(PETHERCAT_SDO_INFO_ENTRY p))
{
    return EC_GET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECSDOINFOENTRY_OFFS_DATATYPE)
                          );
}
EC_INLINE(EC_T_WORD EC_SDOINFOENTRY_GET_BITLEN(PETHERCAT_SDO_INFO_ENTRY p))
{
    return EC_GET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECSDOINFOENTRY_OFFS_BITLEN)
                          );
}
EC_INLINE(EC_T_WORD EC_SDOINFOENTRY_GET_OBJACCESS(PETHERCAT_SDO_INFO_ENTRY p))
{
    return (EC_T_WORD)
           ((EC_GET_FRM_WORD(
                (((EC_T_PBYTE)p)+EC_ECSDOINFOENTRY_OFFS_BITLEN)
                            )>>0)&((1<<6)-1));
}
EC_INLINE(EC_T_WORD EC_SDOINFOENTRY_GET_RXPDOMAPPING(PETHERCAT_SDO_INFO_ENTRY p))
{
    return (EC_T_WORD)
        ((EC_GET_FRM_WORD(
            (((EC_T_PBYTE)p)+EC_ECSDOINFOENTRY_OFFS_BITLEN)
                         )>>6)&1);
}
EC_INLINE(EC_T_WORD EC_SDOINFOENTRY_GET_TXPDOMAPPING(PETHERCAT_SDO_INFO_ENTRY p))
{
    return (EC_T_WORD)
        ((EC_GET_FRM_WORD(
            (((EC_T_PBYTE)p)+EC_ECSDOINFOENTRY_OFFS_BITLEN)
                         )>>7)&1);
}

EC_INLINE(EC_T_VOID EC_SDOINFOENTRY_SET_INDEX(PETHERCAT_SDO_INFO_ENTRY p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECSDOINFOENTRY_OFFS_INDEX),wVal
                   );
}
       
typedef struct TETHERCAT_SDO_INFO_ERROR
{
    EC_T_DWORD      __ErrorCode;
    EC_T_CHAR       ErrorText[1];           /* rest of mailbox data */
} EC_PACKED1 ETHERCAT_SDO_INFO_ERROR, *PETHERCAT_SDO_INFO_ERROR;

EC_INLINE(EC_T_DWORD EC_SDOINFOERROR_GET_ERRORCODE(PETHERCAT_SDO_INFO_ERROR p))
{
    return EC_GET_FRM_DWORD(((EC_T_PBYTE)p));
}
EC_INLINE(EC_T_VOID EC_SDOINFOERROR_SET_ERRORCODE(PETHERCAT_SDO_INFO_ERROR p, EC_T_DWORD dwVal))
{
    EC_SET_FRM_DWORD(((EC_T_PBYTE)p), dwVal);
}

typedef struct TETHERCAT_SDO_INFO_HEADER
{
#ifdef EC_BIG_ENDIAN
    EC_T_BYTE       InComplete      : 1;    /* */
    EC_T_BYTE       OpCode          : 7;    /* == SDO_INFO_TYPE_XXX */  /* 0 */
#else
    EC_T_BYTE       OpCode          : 7;    /* == SDO_INFO_TYPE_XXX */  /* 0 */
    EC_T_BYTE       InComplete      : 1;    /* */
#endif

    EC_T_BYTE       Reserved;               /* == 0 */                  /* 1 */

    EC_T_WORD       __FragmentsLeft;        /* */                       /* 2 */

    union _t_uInfo
    {
        ETHERCAT_SDO_INFO_LIST  List;
        ETHERCAT_SDO_INFO_OBJ   Obj;
        ETHERCAT_SDO_INFO_ENTRY Entry;
        ETHERCAT_SDO_INFO_ERROR Error;
        EC_T_BYTE               Data[1];
    } EC_PACKED1 uInfo;                                                  /* 4 */
} EC_PACKED1 ETHERCAT_SDO_INFO_HEADER, *PETHERCAT_SDO_INFO_HEADER;
#define SDOINFOHDR_CLR(pSdoInfo) OsDbgAssert(sizeof(*pSdoInfo)==sizeof(ETHERCAT_SDO_INFO_HEADER)); OsMemset(pSdoInfo,0,sizeof(ETHERCAT_SDO_INFO_HEADER))

EC_INLINE(EC_T_WORD EC_SDOINFOHDR_GET_FRAGMENTSLEFT(PETHERCAT_SDO_INFO_HEADER p))
{
    return EC_GET_FRM_WORD((((EC_T_PBYTE)p)+2));
}
EC_INLINE(EC_T_VOID EC_SDOINFOHDR_SET_FRAGMENTSLEFT(PETHERCAT_SDO_INFO_HEADER p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD((((EC_T_PBYTE)p)+2), wVal);
}

#define ETHERCAT_SDO_INFO_LISTREQ_LEN   EC_OFFSETOF(ETHERCAT_SDO_INFO_HEADER, uInfo.List.Res)
#define ETHERCAT_SDO_INFO_OBJREQ_LEN    EC_OFFSETOF(ETHERCAT_SDO_INFO_HEADER, uInfo.Obj.Res)
#define ETHERCAT_SDO_INFO_ENTRYREQ_LEN  EC_OFFSETOF(ETHERCAT_SDO_INFO_HEADER, uInfo.Entry.Res)

typedef struct TETHERCAT_EMERGENCY_HEADER
{
    EC_T_WORD   __ErrorCode;
    EC_T_BYTE   ErrorRegister;
    EC_T_BYTE   Data[5];
} EC_PACKED1 ETHERCAT_EMERGENCY_HEADER, *PETHERCAT_EMERGENCY_HEADER;

EC_INLINE(EC_T_WORD EC_EMERGHDR_GET_ERRORCODE(PETHERCAT_EMERGENCY_HEADER p))
{
    return EC_GET_FRM_WORD(((EC_T_PBYTE)p));
}
EC_INLINE(EC_T_VOID EC_EMERGHDR_SET_ERRORCODE(PETHERCAT_EMERGENCY_HEADER p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD(((EC_T_PBYTE)p), wVal);
}

/*---------------------------------------------------------------------------*/
/* FoE (File Access over EtherCAT) */
#define ECAT_FOE_OPCODE_RRQ                 1
#define ECAT_FOE_OPCODE_WRQ                 2
#define ECAT_FOE_OPCODE_DATA                3
#define ECAT_FOE_OPCODE_ACK                 4
#define ECAT_FOE_OPCODE_ERR                 5
#define ECAT_FOE_OPCODE_BUSY                6

#define ECAT_FOE_ERRCODE_NOTDEFINED         0x8000
#define ECAT_FOE_ERRCODE_NOTFOUND           0x8001
#define ECAT_FOE_ERRCODE_ACCESS             0x8002
#define ECAT_FOE_ERRCODE_DISKFULL           0x8003
#define ECAT_FOE_ERRCODE_ILLEAGAL           0x8004
#define ECAT_FOE_ERRCODE_PACKENO            0x8005
#define ECAT_FOE_ERRCODE_EXISTS             0x8006
#define ECAT_FOE_ERRCODE_NOUSER             0x8007
#define ECAT_FOE_ERRCODE_BOOTSTRAPONLY      0x8008
#define ECAT_FOE_ERRCODE_NOTINBOOTSTRAP     0x8009
#define ECAT_FOE_ERRCODE_INVALIDPASSWORD    0x800A
#define ECAT_FOE_ERRCODE_PROGERROR          0x800B

#define EC_ECFOEHDR_OFFS_UHDR               ((EC_T_BYTE)2)
#define EC_ECFOEHDR_OFFS_ENTIRE             ((EC_T_BYTE)4)

typedef struct TETHERCAT_FOE_HEADER
{
    EC_T_BYTE       OpCode;         /* 0 */     /* = 1 (RRQ), = 2 (WRQ), = 3 (DATA), = 4 (ACK), = 5 (ERR), = 6 (BUSY) */
    EC_T_BYTE       Reserved1;      /* 1 */     /* = 0 */
    union _t_uHdr_foeheader
    {
        EC_T_DWORD      __Password; /* 2 */     /* (RRQ, WRQ)       = 0 if unknown */
        EC_T_DWORD      __PacketNo; /* 2 */     /* (DATA, ACK) */
        EC_T_DWORD      __ErrorCode;/* 2 */     /* (ERR) */
        struct _t_sStatus
        {
            EC_T_WORD   __Done;     /* 2 */     /* (BUSY) */
            EC_T_WORD   __Entire;   /* 4 */     /* (BUSY) */
        } EC_PACKED1 sStatus;
    } EC_PACKED1 uHdr;
/*  union */
/*  { */
/*      EC_T_CHAR       Name[]          // (RRQ, WRQ)   rest of mailbox data */
/*      EC_T_BYTE       Data[]          // (DATA)       rest of mailbox data */
/*      EC_T_CHAR       ErrorText[]     // (ERR)        rest of mailbox data */
/*      EC_T_CHAR       BusyText[]      // (BUSY)       rest of mailbox data */
/*  }; */
} EC_PACKED1 ETHERCAT_FOE_HEADER, *PETHERCAT_FOE_HEADER;
#define ETHERCAT_FOE_HEADER_LEN sizeof(ETHERCAT_FOE_HEADER)
#define FOEHDR_CLR(pFoe) OsDbgAssert(sizeof(*pFoe)==(ETHERCAT_FOE_HEADER_LEN)); OsMemset(pFoe,0,(ETHERCAT_FOE_HEADER_LEN))

EC_INLINE(EC_T_DWORD EC_ECFOEHDR_GET_PASSWORD(PETHERCAT_FOE_HEADER p))
{
    return EC_GET_FRM_DWORD( 
        (((EC_T_PBYTE)p)+EC_ECFOEHDR_OFFS_UHDR)
                      );
}
EC_INLINE(EC_T_DWORD EC_ECFOEHDR_GET_PACKETNO(PETHERCAT_FOE_HEADER p))
{
    return EC_GET_FRM_DWORD( 
        (((EC_T_PBYTE)p)+EC_ECFOEHDR_OFFS_UHDR)
        );
}
EC_INLINE(EC_T_DWORD EC_ECFOEHDR_GET_ERRORCODE(PETHERCAT_FOE_HEADER p))
{
    return EC_GET_FRM_DWORD( 
        (((EC_T_PBYTE)p)+EC_ECFOEHDR_OFFS_UHDR)
        );
}
EC_INLINE(EC_T_WORD EC_ECFOEHDR_GET_STATUSDONE(PETHERCAT_FOE_HEADER p))
{
    return EC_GET_FRM_WORD( 
        (((EC_T_PBYTE)p)+EC_ECFOEHDR_OFFS_UHDR)
        );
}
EC_INLINE(EC_T_WORD EC_ECFOEHDR_GET_STATUSENTIRE(PETHERCAT_FOE_HEADER p))
{
    return EC_GET_FRM_WORD( 
        (((EC_T_PBYTE)p)+EC_ECFOEHDR_OFFS_ENTIRE)
        );
}
EC_INLINE(EC_T_VOID EC_ECFOEHDR_SET_ERRORCODE(PETHERCAT_FOE_HEADER p, EC_T_DWORD dwVal))
{
    EC_SET_FRM_DWORD( 
        (((EC_T_PBYTE)p)+EC_ECFOEHDR_OFFS_UHDR),
        dwVal
               );
}
EC_INLINE(EC_T_VOID EC_ECFOEHDR_SET_PACKETNO(PETHERCAT_FOE_HEADER p, EC_T_DWORD dwVal))
{
    EC_SET_FRM_DWORD( 
        (((EC_T_PBYTE)p)+EC_ECFOEHDR_OFFS_UHDR),
        dwVal
               );
}
EC_INLINE(EC_T_VOID EC_ECFOEHDR_SET_PASSWORD(PETHERCAT_FOE_HEADER p, EC_T_DWORD dwVal))
{
    EC_SET_FRM_DWORD( 
        (((EC_T_PBYTE)p)+EC_ECFOEHDR_OFFS_UHDR),
        dwVal
               );
}

#define EC_ECFOEBUSYINFO_OFFS_DONE      ((EC_T_BYTE)0)
#define EC_ECFOEBUSYINFO_OFFS_ENTIRE    ((EC_T_BYTE)2)

typedef struct TETHERCAT_FOE_BUSY_INFO
{
    EC_T_WORD   __Done;         /* 0 */
    EC_T_WORD   __Entire;       /* 2 */
    EC_T_CHAR   Comment[32];    /* 4 */
} EC_PACKED1 ETHERCAT_FOE_BUSY_INFO, *PETHERCAT_FOE_BUSY_INFO;
#define ETHERCAT_FOE_BUSY_INFO_LEN      sizeof(ETHERCAT_FOE_BUSY_INFO)

EC_INLINE(EC_T_WORD EC_ECFOEBUSYINFO_GET_STATUSDONE(PETHERCAT_FOE_BUSY_INFO p))
{
    return EC_GET_FRM_WORD( 
        (((EC_T_PBYTE)p)+EC_ECFOEBUSYINFO_OFFS_DONE)
        );
}
EC_INLINE(EC_T_WORD EC_ECFOEBUSYINFO_GET_STATUSENTIRE(PETHERCAT_FOE_BUSY_INFO p))
{
    return EC_GET_FRM_WORD( 
        (((EC_T_PBYTE)p)+EC_ECFOEBUSYINFO_OFFS_ENTIRE)
        );
}
EC_INLINE(EC_T_VOID EC_ECFOEBUSYINFO_SET_STATUSDONE(PETHERCAT_FOE_BUSY_INFO p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD( 
        (((EC_T_PBYTE)p)+EC_ECFOEBUSYINFO_OFFS_DONE),
        wVal
              );
}
EC_INLINE(EC_T_VOID EC_ECFOEBUSYINFO_SET_STATUSENTIRE(PETHERCAT_FOE_BUSY_INFO p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD( 
        (((EC_T_PBYTE)p)+EC_ECFOEBUSYINFO_OFFS_ENTIRE),
        wVal
              );
}


/*---------------------------------------------------------------------------*/
/* SoE (Servo Profile over EtherCAT) */

#ifdef INCLUDE_SOE_SUPPORT

#define ECAT_SOE_OPCODE_RRQ         1   /*Read request*/
#define ECAT_SOE_OPCODE_RRS         2   /*Read response*/
#define ECAT_SOE_OPCODE_WRQ         3   /*Write request*/
#define ECAT_SOE_OPCODE_WRS         4   /*Write response*/
#define ECAT_SOE_OPCODE_NOTIFY      5   /*Notification*/
#define ECAT_SOE_OPCODE_SLV_INFO    6   /*Slave Info*/
#define ECAT_SOE_OPCODE_RESERVED    7

#define ECAT_SOE_ERRCODE_INVALID_ACCESS     0x0009 /* "Invalid access to element 0" */ 
#define ECAT_SOE_ERRCODE_NOT_EXIST          0x1001 /* "Does not exist" */ 
#define ECAT_SOE_ERRCODE_INVL_ACC_ELEM1     0x1009 /* "Invalid access to element 1" */ 
#define ECAT_SOE_ERRCODE_NAME_NOT_EXIST     0x2001 /* "Name does not exist" */ 
#define ECAT_SOE_ERRCODE_NAME_UNDERSIZE     0x2002 /* "Name undersize in transmission" */ 
#define ECAT_SOE_ERRCODE_NAME_OVERSIZE      0x2003 /* "Name oversize in transmission" */ 
#define ECAT_SOE_ERRCODE_NAME_UNCHANGE      0x2004 /* "Name unchangeable" */ 
#define ECAT_SOE_ERRCODE_NAME_WR_PROT       0x2005 /* "Name currently write-protected" */ 
#define ECAT_SOE_ERRCODE_UNDERS_TRANS       0x3002 /* "Attribute undersize in transmission" */ 
#define ECAT_SOE_ERRCODE_OVERS_TRANS        0x3003 /* "Attribute oversize in transmission" */ 
#define ECAT_SOE_ERRCODE_ATTR_UNCHANGE      0x3004 /* "Attribute unchangeable" */ 
#define ECAT_SOE_ERRCODE_ATTR_WR_PROT       0x3005 /* "Attribute currently write-protected" */ 
#define ECAT_SOE_ERRCODE_UNIT_NOT_EXIST     0x4001 /* "Unit does not exist" */ 
#define ECAT_SOE_ERRCODE_UNIT_UNDERSIZE     0x4002 /* "Unit undersize in transmission" */ 
#define ECAT_SOE_ERRCODE_UNIT_OVERSIZE      0x4003 /* "Unit oversize in transmission" */ 
#define ECAT_SOE_ERRCODE_UNIT_UNCHANGE      0x4004 /* "Unit unchangeable" */ 
#define ECAT_SOE_ERRCODE_UNIT_WR_PROT       0x4005 /* "Unit currently write-protected" */ 
#define ECAT_SOE_ERRCODE_MIN_NOT_EXIST      0x5001 /* "Minimum input value does not exist" */ 
#define ECAT_SOE_ERRCODE_MIN_UNDERSIZE      0x5002 /* "Minimum input value undersize in transmission" */ 
#define ECAT_SOE_ERRCODE_MIN_OVERSIZE       0x5003 /* "Minimum input value oversize in transmission" */ 
#define ECAT_SOE_ERRCODE_MIN_UNCHANGE       0x5004 /* "Minimum input value unchangeable" */ 
#define ECAT_SOE_ERRCODE_MIN_WR_PROT        0x5005 /* "Minimum input value currently write-protected" */ 
#define ECAT_SOE_ERRCODE_MAX_NOT_EXIST      0x6001 /* "Maximum input value does not exist" */ 
#define ECAT_SOE_ERRCODE_MAX_UNDERSIZE      0x6002 /* "Maximum input value undersize in transmission" */ 
#define ECAT_SOE_ERRCODE_MAX_OVERSIZE       0x6003 /* "Maximum input value oversize in transmission" */ 
#define ECAT_SOE_ERRCODE_MAX_UNCHANGE       0x6004 /* "Maximum input value unchangeable" */ 
#define ECAT_SOE_ERRCODE_MAX_WR_PROT        0x6005 /* "Maximum input value currently write-protected" */ 
#define ECAT_SOE_ERRCODE_DATA_NOT_EXIST     0x7001 /* "Data item does not exist" */ 
#define ECAT_SOE_ERRCODE_DATA_UNDERSIZE     0x7002 /* "Data item undersize in transmission" */ 
#define ECAT_SOE_ERRCODE_DATA_OVERSIZE      0x7003 /* "Data item oversize in transmission" */ 
#define ECAT_SOE_ERRCODE_DATA_UNCHANGE      0x7004 /* "Data item unchangeable" */ 
#define ECAT_SOE_ERRCODE_DATA_WR_PROT       0x7005 /* "Data item currently write-protected" */ 
#define ECAT_SOE_ERRCODE_DATA_MIN_LIMIT     0x7006 /* "Data item less than minimum input value limit" */
#define ECAT_SOE_ERRCODE_DATA_MAX_LIMIT     0x7007 /* "Data item exceeds maximum input value limit" */
#define ECAT_SOE_ERRCODE_DATA_INCOR         0x7008 /* "Data item is incorrect" */ 
#define ECAT_SOE_ERRCODE_PASWD_PROT         0x7009 /* "Data item is protected by password" */ 
#define ECAT_SOE_ERRCODE_TEMP_UNCHANGE      0x700A /* "Data item temporary unchangeable (in AT or MDT)" */ 
#define ECAT_SOE_ERRCODE_INVL_INDIRECT      0x700B /* "Invalid indirect" */ 
#define ECAT_SOE_ERRCODE_TEMP_UNCHANGE1     0x700C /* "Data item temporary unchangeable (parameter or opmode...)" */ 
#define ECAT_SOE_ERRCODE_ALREADY_ACTIVE     0x7010 /* "Command already active" */ 
#define ECAT_SOE_ERRCODE_NOT_INTERRUPT      0x7011 /* "Command not interruptable" */ 
#define ECAT_SOE_ERRCODE_CMD_NOT_AVAIL      0x7012 /* "Command not available (in this phase)" */ 
#define ECAT_SOE_ERRCODE_CMD_NOT_AVAIL1     0x7013 /* "Command not available (invalid parameter...)" */ 
#define ECAT_SOE_ERRCODE_NO_DATA            0x7014 /* "No data state"*/
#define ECAT_SOE_ERRCODE_NO_DEFAULT_VALUE   0x8001 /* "No default value */
#define ECAT_SOE_ERRCODE_DEFAULT_LONG       0x8002 /* "Default value transmission too long" */
#define ECAT_SOE_ERRCODE_DEFAULT_WP         0x8004 /* "Default value cannot be changed, read only." */
#define ECAT_SOE_ERRCODE_INVL_DRIVE_NO      0x800A /* "Invalid drive number" */
#define ECAT_SOE_ERRCODE_GENERAL_ERROR      0x800B /* "General error" */
#define ECAT_SOE_ERRCODE_NO_ELEM_ADR        0x800C /* "No element addressed" */




#define EC_ECSOEHDR_OFFS_IDN      ((EC_T_BYTE)2)
#define EC_ECSOEHDR_OFFS_FRAGLEFT ((EC_T_BYTE)2)
typedef struct TETHERCAT_SOE_HEADER
{
#ifdef EC_BIG_ENDIAN
    EC_T_BYTE       DriveNo     : 3;
    EC_T_BYTE       Error       : 1;
    EC_T_BYTE       Incomplete  : 1;
    EC_T_BYTE       OpCode      : 3;
#else
    EC_T_BYTE       OpCode      : 3;
    EC_T_BYTE       Incomplete  : 1;
    EC_T_BYTE       Error       : 1;
    EC_T_BYTE       DriveNo     : 3;
#endif
    EC_T_BYTE       byElements;
    union _t_uIdnFragLeft
    {
        EC_T_WORD   wIdn;
        EC_T_WORD   wFragmentsLeft;
    } EC_PACKED1 uIdnFragLeft;
} EC_PACKED1 ETHERCAT_SOE_HEADER, *PETHERCAT_SOE_HEADER;


typedef struct TETHERCAT_SOE_EMERGENCY_HEADER
{
    EC_T_BYTE   Data[64];
} EC_PACKED1 ETHERCAT_SOE_EMERGENCY_HEADER, *PETHERCAT_SOE_EMERGENCY_HEADER;

typedef struct TETHERCAT_SOE_NOTIFICATION_HEADER
{
    EC_T_WORD   DataStatus;
} EC_PACKED1 ETHERCAT_SOE_NOTIFICATION_HEADER, *PETHERCAT_SOE_NOTIFICATION_HEADER;



/* definition of bitmask for above element flags */
#ifdef EC_BIG_ENDIAN
#define SOE_BM_ELEMENTFLAG_DATATSTATE   0x80
#define SOE_BM_ELEMENTFLAG_NAME         0x40
#define SOE_BM_ELEMENTFLAG_ATTRIBUTE    0x20
#define SOE_BM_ELEMENTFLAG_UNIT         0x10
#define SOE_BM_ELEMENTFLAG_MIN          0x08
#define SOE_BM_ELEMENTFLAG_MAX          0x04
#define SOE_BM_ELEMENTFLAG_VALUE        0x02
#define SOE_BM_ELEMENTFLAG_DEFAULT      0x01
#else
#define SOE_BM_ELEMENTFLAG_DATATSTATE   0x01
#define SOE_BM_ELEMENTFLAG_NAME         0x02
#define SOE_BM_ELEMENTFLAG_ATTRIBUTE    0x04
#define SOE_BM_ELEMENTFLAG_UNIT         0x08
#define SOE_BM_ELEMENTFLAG_MIN          0x10
#define SOE_BM_ELEMENTFLAG_MAX          0x20
#define SOE_BM_ELEMENTFLAG_VALUE        0x40
#define SOE_BM_ELEMENTFLAG_DEFAULT      0x80
#endif

#define SOE_HDR_IDN_OFFSET              (sizeof(EC_T_BYTE)+sizeof(EC_T_BYTE))                   /* OpCode/... + DataState/... */
#define SOE_HDR_DATA_OFFSET             (sizeof(EC_T_BYTE)+sizeof(EC_T_BYTE)+sizeof(EC_T_WORD)) /* OpCode/... + DataState/... + IDN/FragLeft */
#define ETHERCAT_SOE_HEADER_LEN         sizeof(ETHERCAT_SOE_HEADER)
#define ETHERCAT_MIN_SOE_MBOX_LEN       (ETHERCAT_MBOX_HEADER_LEN + ETHERCAT_SOE_HEADER_LEN)
#define SOEHDR_CLR(pSoe) OsDbgAssert(sizeof(*pSoe)==ETHERCAT_SOE_HEADER_LEN); OsMemset(pSoe,0,ETHERCAT_SOE_HEADER_LEN)

EC_INLINE(EC_T_WORD EC_ECSOEHDR_GET_IDN(PETHERCAT_SOE_HEADER p))
{
    return EC_GET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECSOEHDR_OFFS_IDN)
                     );
}
EC_INLINE(EC_T_WORD EC_ECSOEHDR_GET_FRAGLEFT(PETHERCAT_SOE_HEADER p))
{
    return EC_GET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECSOEHDR_OFFS_FRAGLEFT)
                     );
}
EC_INLINE(EC_T_DWORD EC_ECSOEHDR_GET_ERRORCODE(PETHERCAT_SOE_HEADER p))
{
    return EC_GET_FRM_WORD( 
        (((EC_T_PBYTE)p)+SOE_HDR_DATA_OFFSET)
        );
}



EC_INLINE(EC_T_VOID EC_ECSOEHDR_SET_IDN(PETHERCAT_SOE_HEADER p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECSOEHDR_OFFS_IDN),
        wVal
              );
}
EC_INLINE(EC_T_VOID EC_ECSOEHDR_SET_FRAGLEFT(PETHERCAT_SOE_HEADER p, EC_T_WORD wVal))
{
    EC_SET_FRM_WORD(
        (((EC_T_PBYTE)p)+EC_ECSOEHDR_OFFS_FRAGLEFT),
        wVal
              );
}


#endif /* #ifdef INCLUDE_SOE_SUPPORT */

/*---------------------------------------------------------------------------*/
#define EC_ECMBXCMDDESC_OFFS_TRANSITION         ((EC_T_BYTE)0)
#define EC_ECMBXCMDDESC_OFFS_PROTOCOL           ((EC_T_BYTE)2)
#define EC_ECMBXCMDDESC_OFFS_DATALEN            ((EC_T_BYTE)4)
#define EC_ECMBXCMDDESC_OFFS_CMTLEN             ((EC_T_BYTE)8)
#define EC_ECMBXCMDDESC_OFFS_MBXCMDTIMEOUT      ((EC_T_BYTE)10)
#define EC_ECMBXCMDDESC_OFFS_RETRIES            ((EC_T_BYTE)12)
#define EC_ECMBXCMDDESC_OFFS_FOENAMESIZE        ((EC_T_BYTE)32)
#define EC_ECMBXCMDDESC_OFFS_FOEPASSWORD        ((EC_T_BYTE)36)

typedef struct TEcMailboxCmdDesc
{
    EC_T_WORD                   __transition;       /*  0 */
    EC_T_WORD                   __protocol;         /*  2 */   
    EC_T_DWORD                  __dataLen;          /*  4 */
    EC_T_WORD                   __cmtLen;           /*  8 */    /* (excl. \0) */
    EC_T_WORD                   __wMbxCmdTimeout;   /* 10 */    /* in ms */
    EC_T_WORD                   __retries;          /* 12 */
    EC_T_WORD                   reserved1;          /* 14 */
    EC_T_DWORD                  reserved2[4];       /* 16 */
    union _t_uMbxHdr                                
    {                                               
        struct _t_sCoe                              
        {                                           
            ETHERCAT_SDO_HEADER EcSdoHeader;        /* 32 */
            EC_T_BYTE           data[1];            /* 32 + ETHERCAT_SDO_HEADER_LEN */    
                                                                /* data[dataLen-sizeof(sdo)]; */
        } EC_PACKED1 coe;
        struct _t_sEoe
        {
            EC_T_BYTE           data[1];            /* 32 + ETHERCAT_EOE_HEADER_LEN */    
                                                                /* data[dataLen-sizeof(sdo)]; */
        } EC_PACKED1 eoe;                            
#if (defined INCLUDE_FOE_SUPPORT)
        struct _t_sFoe                              
        {                                           
            ETHERCAT_FOE_HEADER EcFoeHeader;        /* 32 */
            EC_T_CHAR           name[1];            /* 40 */    /* no \0 */
        /*  EC_T_BYTE           data[];     */
        } EC_PACKED1 foe;
#endif
#ifdef INCLUDE_SOE_SUPPORT
        struct _t_sSoe
        {                                           
            ETHERCAT_SOE_HEADER EcSoeHeader;        /* 32 */
            EC_T_BYTE           data[1];            /* 32 + ETHERCAT_SDO_HEADER_LEN */    
                                                                /* data[dataLen-sizeof(sdo)]; */
        } EC_PACKED1 soe;
#endif
        EC_T_BYTE               data[1];            /* 32 */    /* data[dataLen]; */
    } EC_PACKED1 uMbxHdr;
/*  EC_T_CHAR   cmt[cmtLen+1]; */
} EC_PACKED1 EcMailboxCmdDesc, *PEcMailboxCmdDesc;

EC_INLINE(EC_T_WORD EC_ECMBXCMDDESC_GET_TRANSITION(PEcMailboxCmdDesc p))
{
    return EC_GETWORD( 
        (((EC_T_PBYTE)p)+EC_ECMBXCMDDESC_OFFS_TRANSITION)
                     );
}
EC_INLINE(EC_T_WORD EC_ECMBXCMDDESC_GET_PROTOCOL(PEcMailboxCmdDesc p))
{
    return EC_GETWORD( 
        (((EC_T_PBYTE)p)+EC_ECMBXCMDDESC_OFFS_PROTOCOL)
                     );
}
EC_INLINE(EC_T_DWORD EC_ECMBXCMDDESC_GET_DATALEN(PEcMailboxCmdDesc p))
{
    return EC_GETDWORD( 
        (((EC_T_PBYTE)p)+EC_ECMBXCMDDESC_OFFS_DATALEN)
                      );
}
EC_INLINE(EC_T_WORD EC_ECMBXCMDDESC_GET_CMTLEN(PEcMailboxCmdDesc p))
{
    return EC_GETWORD( 
        (((EC_T_PBYTE)p)+EC_ECMBXCMDDESC_OFFS_CMTLEN)
                     );
}
EC_INLINE(EC_T_WORD EC_ECMBXCMDDESC_GET_MBXTIMEOUT(PEcMailboxCmdDesc p))
{
    return EC_GETWORD( 
        (((EC_T_PBYTE)p)+EC_ECMBXCMDDESC_OFFS_MBXCMDTIMEOUT)
                     );
}
EC_INLINE(EC_T_WORD EC_ECMBXCMDDESC_GET_RETRIES(PEcMailboxCmdDesc p))
{
    return EC_GETWORD( 
        (((EC_T_PBYTE)p)+EC_ECMBXCMDDESC_OFFS_RETRIES)
        );
}
EC_INLINE(EC_T_WORD EC_ECMBXCMDDESC_GET_FOENAMESIZE(PEcMailboxCmdDesc p))
{
    return EC_GETWORD( 
        (((EC_T_PBYTE)p)+EC_ECMBXCMDDESC_OFFS_FOENAMESIZE)
                     );
}
EC_INLINE(EC_T_DWORD EC_ECMBXCMDDESC_GET_FOEPASSWORD(PEcMailboxCmdDesc p))
{
    return EC_GETDWORD( 
        (((EC_T_PBYTE)p)+EC_ECMBXCMDDESC_OFFS_FOEPASSWORD)
        );
}
EC_INLINE(EC_T_VOID EC_ECMBXCMDDESC_SET_DATALEN(PEcMailboxCmdDesc p, EC_T_DWORD dwVal))
{
    EC_SETDWORD( 
        (((EC_T_PBYTE)p)+EC_ECMBXCMDDESC_OFFS_DATALEN),
        dwVal
               );
}
EC_INLINE(EC_T_VOID EC_ECMBXCMDDESC_SET_CMTLEN(PEcMailboxCmdDesc p, EC_T_WORD wVal))
{
    EC_SETWORD( 
        (((EC_T_PBYTE)p)+EC_ECMBXCMDDESC_OFFS_CMTLEN),
        wVal
              );
}
EC_INLINE(EC_T_VOID EC_ECMBXCMDDESC_SET_PROTOCOL(PEcMailboxCmdDesc p, EC_T_WORD wVal))
{
    EC_SETWORD( 
        (((EC_T_PBYTE)p)+EC_ECMBXCMDDESC_OFFS_PROTOCOL),
        wVal
              );
}
EC_INLINE(EC_T_VOID EC_ECMBXCMDDESC_SET_TRANSITION(PEcMailboxCmdDesc p, EC_T_WORD wVal))
{
    EC_SETWORD( 
        (((EC_T_PBYTE)p)+EC_ECMBXCMDDESC_OFFS_TRANSITION),
        wVal
              );
}
EC_INLINE(EC_T_VOID EC_ECMBXCMDDESC_SET_MBXTIMEOUT(PEcMailboxCmdDesc p, EC_T_WORD wVal))
{
    EC_SETWORD( 
        (((EC_T_PBYTE)p)+EC_ECMBXCMDDESC_OFFS_MBXCMDTIMEOUT),
        wVal
              );
}

#define SIZEOF_EcMailboxCmdDesc(p)  (EC_OFFSETOF(EcMailboxCmdDesc, uMbxHdr.data) + EC_ECMBXCMDDESC_GET_DATALEN(((PEcMailboxCmdDesc)(p))) + EC_ECMBXCMDDESC_GET_CMTLEN(((PEcMailboxCmdDesc)(p))) + 1)
#define NEXT_EcMailboxCmdDesc(p)    (PEcMailboxCmdDesc)&(((EC_T_BYTE*)(p))[SIZEOF_EcMailboxCmdDesc(p)])
#define EcMailboxCmdDescComment(p)  (EC_T_CHAR*)&(((EC_T_BYTE*)(p))[EC_OFFSETOF(EcMailboxCmdDesc, uMbxHdr.data) + EC_ECMBXCMDDESC_GET_DATALEN(((PEcMailboxCmdDesc)(p)))])
#define SIZEOF_EcMailboxCmdDescCoeData(p)   (EC_ECMBXCMDDESC_GET_DATALEN(((PEcMailboxCmdDesc)(p)))-sizeof(ETHERCAT_SDO_HEADER))
#define SIZEOF_EcMailboxCmdDescEoeData(p)   (EC_ECMBXCMDDESC_GET_DATALEN(((PEcMailboxCmdDesc)(p))))
#ifdef INCLUDE_SOE_SUPPORT
#define SIZEOF_EcMailboxCmdDescSoeData(p)   (EC_ECMBXCMDDESC_GET_DATALEN(((PEcMailboxCmdDesc)(p)))-sizeof(ETHERCAT_SOE_HEADER))
#endif

/*---------------------------------------------------------------------------*/

#define EC_ECETHCRESW_OFFS_MAXPORTS     ((EC_T_BYTE)0)
#define EC_ECETHCRESW_OFFS_MAXFRAMES    ((EC_T_BYTE)4)
#define EC_ECETHCRESW_OFFS_MAXMACS      ((EC_T_BYTE)8)
#define EC_ECETHCRESW_OFFS_SWITCHTONIC  ((EC_T_BYTE)12)

typedef struct TEthernetCreateSwitch
{
    EC_T_DWORD      __maxPorts;     /*  0 */
    EC_T_DWORD      __maxFrames;    /*  4 */
    EC_T_DWORD      __maxMACs;      /*  8 */
    EC_T_BOOL       __switchToNIC;  /* 12 */
    EC_T_DWORD      reserved[4];    /* 16 */
} EC_PACKED1 EthernetCreateSwitch, *PEthernetCreateSwitch;

EC_INLINE(EC_T_DWORD EC_ECETHCRESW_GET_MAXPORTS(PEthernetCreateSwitch p))
{
    return EC_GETDWORD( 
        (((EC_T_PBYTE)p)+EC_ECETHCRESW_OFFS_MAXPORTS)
                      );
}
EC_INLINE(EC_T_DWORD EC_ECETHCRESW_GET_MAXFRAMES(PEthernetCreateSwitch p))
{
    return EC_GETDWORD( 
        (((EC_T_PBYTE)p)+EC_ECETHCRESW_OFFS_MAXFRAMES)
        );
}
EC_INLINE(EC_T_DWORD EC_ECETHCRESW_GET_MAXMACS(PEthernetCreateSwitch p))
{
    return EC_GETDWORD( 
        (((EC_T_PBYTE)p)+EC_ECETHCRESW_OFFS_MAXMACS)
        );
}
EC_INLINE(EC_T_BOOL EC_ECETHCRESW_GET_SWITCHTONIC(PEthernetCreateSwitch p))
{
    return (EC_TRUE == 
            EC_GETDWORD( 
            (((EC_T_PBYTE)p)+EC_ECETHCRESW_OFFS_SWITCHTONIC)
                       )
           );
}
EC_INLINE(EC_T_VOID EC_ECETHCRESW_SET_MAXPORTS(PEthernetCreateSwitch p, EC_T_DWORD dwVal))
{
    EC_SETDWORD( 
        (((EC_T_PBYTE)p)+EC_ECETHCRESW_OFFS_MAXPORTS),
        dwVal
               );
}
EC_INLINE(EC_T_VOID EC_ECETHCRESW_SET_MAXFRAMES(PEthernetCreateSwitch p, EC_T_DWORD dwVal))
{
    EC_SETDWORD( 
        (((EC_T_PBYTE)p)+EC_ECETHCRESW_OFFS_MAXFRAMES),
        dwVal
               );
}
EC_INLINE(EC_T_VOID EC_ECETHCRESW_SET_MAXMACS(PEthernetCreateSwitch p, EC_T_DWORD dwVal))
{
    EC_SETDWORD( 
        (((EC_T_PBYTE)p)+EC_ECETHCRESW_OFFS_MAXMACS),
        dwVal
               );
}
EC_INLINE(EC_T_VOID EC_ECETHCRESW_SET_SWITCHTONIC(PEthernetCreateSwitch p, EC_T_BOOL bVal))
{
    EC_SETDWORD( 
        (((EC_T_PBYTE)p)+EC_ECETHCRESW_OFFS_SWITCHTONIC),
        bVal
               );
}

typedef struct _EC_T_EEP_REGS
{
    EC_T_WORD   wCtrlStatus;                /* 0x0502 */
    EC_T_DWORD  dwEEPAddress;               /* 0x0504 */
    EC_T_DWORD  dwEEPData;                  /* 0x0508 */
} EC_PACKED1 EC_T_SB_EEP_REGS, *EC_PT_SB_EEP_REGS;


/*-PACK SETTINGS-------------------------------------------------------------*/
#if defined  __GNUC__   /* GNU */

#undef  EC_PACKED1

#if (defined __RCX__)
#pragma pack ()
#endif

#elif defined _MSC_VER  /* MICROSOFT */

#undef  EC_PACKED1
#pragma pack (pop)

#elif defined __MET__

#undef EC_PACKED1
/*#pragma pack (pop)*/
#pragma Pop_align_members()

#else
#endif

/*-HELPER FUNCTIONS-----------------------------------------------------------*/
extern char* GetStateChangeNameShort(EC_T_WORD transition);


#endif /* INC_ETHERNETSERVICES */


/*-END OF SOURCE FILE--------------------------------------------------------*/
